module "twitch_networking" {
  source      = "../twitch_networking"
  environment = var.environment
}

data "aws_caller_identity" "current" {
}

data "aws_iam_policy" "s2s_assume_role_policy" {
  arn = "arn:aws:iam::${data.aws_caller_identity.current.account_id}:policy/S2SAssumeRole"
}

resource "aws_iam_role" "controlplane_updater" {
  name = "controlplane-updater"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "Service": "lambda.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF

}

// Allow controlplane-updater to write logs to CloudWatch Logs and access VPC resources
resource "aws_iam_policy" "controlplane_updater_policy" {
  name        = "controlplane-updater-policy"
  path        = "/"
  description = "Used by the controlplane-updater lambda for logging and vpc access"

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": [
                "logs:CreateLogGroup",
                "logs:CreateLogStream",
                "logs:PutLogEvents",
                "ec2:CreateNetworkInterface",
                "ec2:DescribeNetworkInterfaces",
                "ec2:DeleteNetworkInterface"
            ],
            "Resource": "*"
        }
    ]
}
EOF

}

resource "aws_iam_role_policy_attachment" "s2s_assume_role_attachment" {
  role       = aws_iam_role.controlplane_updater.id
  policy_arn = data.aws_iam_policy.s2s_assume_role_policy.arn
}

resource "aws_iam_role_policy_attachment" "controlplane_updater_policy_attachment" {
  role       = aws_iam_role.controlplane_updater.name
  policy_arn = aws_iam_policy.controlplane_updater_policy.arn
}

// This role can be assumed by Jenkins in the devtools account.
// Used to invoke the controlplane-updater lambda
resource "aws_iam_role" "controlplane_updater_jenkins" {
  name = "controlplane-updater-jenkins"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Principal": {
        "AWS": "arn:aws:iam::043714768218:role/jenkins-prod-slave"
      },
      "Action": "sts:AssumeRole",
      "Condition": {}
    },
    {
      "Effect": "Allow",
      "Principal": {
        "AWS": "arn:aws:iam::${var.account_id}:user/jenkins"
      },
      "Action": "sts:AssumeRole",
      "Condition": {}
    }
  ]
}
EOF

}

resource "aws_iam_policy" "controlplane_updater_jenkins_policy" {
  name        = "controlplane-updater-jenkins-policy"
  path        = "/"
  description = "Allows Jenkins to assume-role and invoke controlplane-updater lambda"

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": [
                "lambda:InvokeFunction",
                "lambda:UpdateFunctionCode"
            ],
            "Resource": [
                "${aws_lambda_function.controlplane_updater.arn}"
            ]
        }
    ]
}
EOF

}

resource "aws_iam_role_policy_attachment" "controlplane_updater_jenkins_policy_attachment" {
  role       = aws_iam_role.controlplane_updater_jenkins.name
  policy_arn = aws_iam_policy.controlplane_updater_jenkins_policy.arn
}

resource "aws_lambda_function" "controlplane_updater" {
  filename      = "${path.module}/lambda-hello-world.zip"
  function_name = "controlplane-updater"
  role          = aws_iam_role.controlplane_updater.arn
  handler       = "controlplane-updater"
  runtime       = "go1.x"
  timeout       = 60

  vpc_config {
    subnet_ids         = module.twitch_networking.private_subnets
    security_group_ids = [module.twitch_networking.twitch_subnets_sg_id]
  }

  environment {
    variables = {
      ENVIRONMENT = var.environment
    }
  }

  lifecycle {
    ignore_changes = [filename]
  }
}

