resource "aws_s3_bucket" "cloudformation" {
  bucket = var.environment == "prod" ? "eventbus-setup" : "eventbus-${var.environment}-setup"
  acl    = "private"

  versioning {
    enabled = true
  }

  logging {
    target_bucket = aws_s3_bucket.cloudformation_log.id
    target_prefix = "log/"
  }
}

resource "aws_s3_bucket" "cloudformation_log" {
  bucket = "${var.environment == "prod" ? "eventbus-setup" : "eventbus-${var.environment}-setup"}-log"
  acl    = "log-delivery-write"
}

data "template_file" "cloudformation_versions" {
  for_each = toset(local.versions)
  template = file("${path.module}/cloudformation/cloudformation_${each.key}.yaml.tpl")

  vars = {
    event_bus_account_id          = var.account_id
    iam_suffix                    = var.environment == "prod" ? "" : "-${var.environment}"
    s3_bucket                     = var.s3_bucket
    encryption_at_rest_cmk_key_id = var.encryption_at_rest_cmk_key_id
    authorized_fields_cmk_key_id  = var.authorized_fields_cmk_key_id
    cloudformation_version        = each.key
  }
}

resource "aws_s3_bucket_object" "cloudformation_versions" {
  for_each = toset(local.versions)

  bucket       = aws_s3_bucket.cloudformation.id
  key          = "cloudformation_${each.key}.yaml"
  content      = data.template_file.cloudformation_versions[each.key].rendered
  content_type = "text/plain"
  etag = md5(data.template_file.cloudformation_versions[each.key].rendered)
  acl = "public-read"
}

resource "aws_s3_bucket_object" "cloudformation_latest" {
  bucket       = aws_s3_bucket.cloudformation.id
  key          = "cloudformation.yaml"
  content      = data.template_file.cloudformation_versions[var.latest_version].rendered
  content_type = "text/plain"
  etag         = md5(data.template_file.cloudformation_versions[var.latest_version].rendered)
  acl          = "public-read"
}
