AWSTemplateFormatVersion: "2010-09-09"
Description: Event Bus Cross-Account

Resources:

  # This role is assumed by the Event Bus account.
  #
  # Allows:
  # 1) Creating/modifying/deleting subscriptions to SNS topics in the Event Bus account with "eventbus" in the name
  # 2) Creating/modifying/deleting SQS queues in your account with "eventbus" in the name
  EventBusRole:
    Type: AWS::IAM::Role
    Properties:
      Path: "/EventBus${iam_suffix}/"
      RoleName: "EventBus${iam_suffix}"
      AssumeRolePolicyDocument:
        Statement:
          - Effect: "Allow"
            Principal:
              AWS: "arn:aws:iam::${event_bus_account_id}:root"
            Action:
              - "sts:AssumeRole"
      Policies:
        - PolicyName: "EventBusSNS${iam_suffix}"
          PolicyDocument:
            Version: "2012-10-17"
            Statement:
              - Effect: "Allow"
                Action:
                  - "sns:Subscribe"
                  - "sns:ConfirmSubscription"
                Resource:
                  - "arn:aws:sns:*:${event_bus_account_id}:*eventbus*"
              - Effect: "Allow"
                Action:
                  - "sns:Unsubscribe"
                Resource: "*"

        - PolicyName: "EventBusSQS${iam_suffix}"
          PolicyDocument:
            Version: "2012-10-17"
            Statement:
              - Effect: "Allow"
                Action:
                  - "sqs:CreateQueue"
                  - "sqs:DeleteQueue"
                  - "sqs:AddPermission"
                  - "sqs:RemovePermission"
                  - "sqs:GetQueueAttributes"
                  - "sqs:SetQueueAttributes"
                  - "sqs:TagQueue"
                  - "sqs:UntagQueue"
                  - "sqs:ListQueueTags"
                  - "sqs:GetQueueUrl"
                Resource:
                  - !Sub "arn:aws:sqs:*:$${AWS::AccountId}:*eventbus*"

  # This policy should be attached to application roles that run the Event Bus client library.
  #
  # Allows:
  # 1) Assuming roles named "EventBusPublisherV*"
  # 2) Assuming roles named "EventBusSubscriberV*"
  #
  # Temporarily allows:
  # 1) Publishing to SNS topics in the Event Bus account with "eventbus" in the name
  # 2) Processing messages from SQS queues in your account with "eventbus" in the name
  # (will be removed once the Event Bus client has been upgraded to use assume roles)
  EventBusAccessPolicy:
    Type: AWS::IAM::ManagedPolicy
    Properties:
      Path: "/EventBus${iam_suffix}/"
      ManagedPolicyName: "EventBusAccess${iam_suffix}"
      Description: "Provides access to the Event Bus"
      PolicyDocument:
        Version: "2012-10-17"
        Statement:
          - Effect: "Allow"
            Action:
              - "sts:GetCallerIdentity"
            Resource: "*"

          - Effect: "Allow"
            Action:
              - "sts:AssumeRole"
            Resource:
              - !Sub "arn:aws:iam::$${AWS::AccountId}:role/EventBus${iam_suffix}/EventBusPublisherV*"
              - !Sub "arn:aws:iam::$${AWS::AccountId}:role/EventBus${iam_suffix}/EventBusSubscriberV*"

          - Effect: "Allow"
            Action:
              - "sns:Publish"
            Resource:
              - "arn:aws:sns:*:${event_bus_account_id}:*eventbus*"

          - Effect: "Allow"
            Action:
              - "sqs:ReceiveMessage"
              - "sqs:ChangeMessageVisibility"
              - "sqs:DeleteMessage"
              - "sqs:GetQueueUrl"
              - "sqs:GetQueueAttributes"
              - "sqs:ListQueueTags"
            Resource:
              - !Sub "arn:aws:sqs:*:$${AWS::AccountId}:*eventbus*"

          - Effect: "Allow"
            Action:
              - "kms:GenerateDataKey*"
              - "kms:Decrypt"
            Resource:
                - "arn:aws:kms:*:${event_bus_account_id}:key/${encryption_at_rest_cmk_key_id}"

          - Effect: "Allow"
            Action:
              - "kms:GenerateDataKey*"
              - "kms:Decrypt"
            Resource:
                - "arn:aws:kms:*:${event_bus_account_id}:key/${authorized_fields_cmk_key_id}"

  # This role is assumed by an application that is using the Event Bus client library.
  #
  # Allows:
  # 1) Reading objects from S3 in "${s3_bucket}/publishers/*"
  # 2) Publishing to SNS topics in the Event Bus account with "eventbus" in the name
  EventBusPublisherRoleV1:
    Type: AWS::IAM::Role
    Properties:
      Path: "/EventBus${iam_suffix}/"
      RoleName: "EventBusPublisherV1${iam_suffix}"
      AssumeRolePolicyDocument:
        Statement:
          - Effect: "Allow"
            Principal:
              AWS: !Sub "arn:aws:iam::$${AWS::AccountId}:root"
            Action:
              - "sts:AssumeRole"
      Policies:
        - PolicyName: "EventBusPublisherV1S3${iam_suffix}"
          PolicyDocument:
            Version: "2012-10-17"
            Statement:
              - Effect: "Allow"
                Action:
                  - "s3:GetObject"
                Resource:
                  - "arn:aws:s3:::${s3_bucket}/publishers/*"

        - PolicyName: "EventBusPublisherV1SNS${iam_suffix}"
          PolicyDocument:
            Version: "2012-10-17"
            Statement:
              - Effect: "Allow"
                Action:
                  - "sns:Publish"
                Resource:
                  - "arn:aws:sns:*:${event_bus_account_id}:*eventbus*"

        - PolicyName: "EventBusPublisherV1KMS${iam_suffix}"
          PolicyDocument:
            Version: "2012-10-17"
            Statement:
              - Effect: "Allow"
                Action:
                  - "kms:GenerateDataKey*"
                  - "kms:Decrypt"
                Resource:
                  - "arn:aws:kms:*:${event_bus_account_id}:key/${encryption_at_rest_cmk_key_id}"

  # This role is assumed by an application that is using the Event Bus client library.
  #
  # Allows:
  # 1) Reading objects from S3 in "${s3_bucket}/subscribers/*"
  # 2) Processing messages from SQS queues in your account with "eventbus" in the name
  EventBusSubscriberRoleV1:
    Type: AWS::IAM::Role
    Properties:
      Path: "/EventBus${iam_suffix}/"
      RoleName: "EventBusSubscriberV1${iam_suffix}"
      AssumeRolePolicyDocument:
        Statement:
          - Effect: "Allow"
            Principal:
              AWS: !Sub "arn:aws:iam::$${AWS::AccountId}:root"
            Action:
              - "sts:AssumeRole"
      Policies:
        - PolicyName: "EventBusSubscriberV1S3${iam_suffix}"
          PolicyDocument:
            Version: "2012-10-17"
            Statement:
              - Effect: "Allow"
                Action:
                  - "s3:GetObject"
                Resource:
                  - "arn:aws:s3:::${s3_bucket}/subscribers/*"

        - PolicyName: "EventBusSubscriberV1SQS${iam_suffix}"
          PolicyDocument:
            Version: "2012-10-17"
            Statement:
              - Effect: "Allow"
                Action:
                  - "sqs:ReceiveMessage"
                  - "sqs:ChangeMessageVisibility"
                  - "sqs:DeleteMessage"
                  - "sqs:GetQueueUrl"
                  - "sqs:GetQueueAttributes"
                  - "sqs:ListQueueTags"
                Resource:
                  - !Sub "arn:aws:sqs:*:$${AWS::AccountId}:*eventbus*"

        - PolicyName: "EventBusSubscriberV1KMS${iam_suffix}"
          PolicyDocument:
            Version: "2012-10-17"
            Statement:
              - Effect: "Allow"
                Action:
                  - "kms:GenerateDataKey*"
                  - "kms:Decrypt"
                Resource:
                  - "arn:aws:kms:*:${event_bus_account_id}:key/${encryption_at_rest_cmk_key_id}"

Outputs:
  EventBusAccessPolicyARN:
    Description: "EventBusAccess policy ARN"
    Value: !Ref EventBusAccessPolicy
    Export:
      Name: "eventbus-access-policy-arn${iam_suffix}"

  EventBusKMSMasterKeyARN:
    Description: "Event Bus KMS key ARN"
    Value: "arn:aws:kms:*:${event_bus_account_id}:key/${encryption_at_rest_cmk_key_id}"
    Export:
      Name: "eventbus-kms-master-key-arn${iam_suffix}"

  EventBusSQSPolicyDocument:
    Description: "SQS policy document to provide the Event Bus SendMessage permissions"
    Value: |
      {
        "Version": "2012-10-17",
        "Statement": [{
          "Effect": "Allow",
          "Principal": "*",
          "Action": "sqs:SendMessage",
          "Resource": "*",
          "Condition":{
            "ArnEquals":{
              "aws:SourceArn": "arn:aws:*:*:${event_bus_account_id}:*"
            }
          }
        }]
      }
    Export:
      Name: "eventbus-sqs-policy-document${iam_suffix}"
