resource "aws_alb" "controlplane_httpserver" {
  name            = var.controlplane_httpserver_app_name
  internal        = true
  subnets         = module.twitch_networking.private_subnets
  security_groups = [module.twitch_networking.twitch_subnets_sg_id]
}

resource "aws_alb_target_group" "controlplane_httpserver" {
  deregistration_delay = 5
  slow_start           = var.lb_slow_start_seconds
  name                 = var.controlplane_httpserver_app_name
  protocol             = "HTTP"
  vpc_id               = module.twitch_networking.vpc_id

  # The port is required, but overriden by ECS. Using a placeholder here.
  port = "1"

  health_check {
    protocol            = "HTTP"
    path                = "/health"
    healthy_threshold   = 3
    unhealthy_threshold = 2
    timeout             = 2
    interval            = 5
    matcher             = 200
  }
}

# Redirect all traffic from the ALB to the target group
resource "aws_alb_listener" "controlplane_httpserver_http" {
  load_balancer_arn = aws_alb.controlplane_httpserver.id
  port              = "80"
  protocol          = "HTTP"

  default_action {
    target_group_arn = aws_alb_target_group.controlplane_httpserver.id
    type             = "forward"
  }
}

resource "aws_alb_listener" "controlplane_httpserver_https" {
  load_balancer_arn = aws_alb.controlplane_httpserver.id
  port              = "443"
  protocol          = "HTTPS"
  certificate_arn   = aws_acm_certificate.controlplane_ssl_cert_a2z.arn

  default_action {
    target_group_arn = aws_alb_target_group.controlplane_httpserver.id
    type             = "forward"
  }
}

# Deprecated certificate, used as optional during migration
# TODO: remove after cutover is complete
resource "aws_lb_listener_certificate" "deprecated_cert_controlplane" {
  listener_arn    = "${aws_alb_listener.controlplane_httpserver_https.arn}"
  certificate_arn = "${aws_acm_certificate.controlplane_httpserver_ssl_cert.arn}"
}


# Request an SSL certificate through ACM.
# To validate a certificate, create the CNAME DNS record required by the DNS validation method
# in the internal DNS dashboard. See https://wiki.twitch.com/x/CoFQB.
resource "aws_acm_certificate" "controlplane_httpserver_ssl_cert" {
  domain_name       = "eventbus-controlplane-${var.environment}.internal.justin.tv"
  validation_method = "DNS"

  tags = {
    app = var.controlplane_httpserver_app_name
    env = var.environment
  }

  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_cloudwatch_log_group" "controlplane_httpserver_logs" {
  name              = "${var.controlplane_httpserver_app_name}-logs"
  retention_in_days = var.app_log_retention_days

  tags = {
    app = var.controlplane_httpserver_app_name
    env = var.environment
  }
}

resource "aws_cloudwatch_log_group" "controlplane_converger_logs" {
  name              = "${var.controlplane_converger_app_name}-logs"
  retention_in_days = var.app_log_retention_days

  tags = {
    app = var.controlplane_converger_app_name
    env = var.environment
  }
}

resource "aws_ecs_task_definition" "controlplane_httpserver" {
  family             = var.controlplane_httpserver_app_name
  network_mode       = "bridge"
  execution_role_arn = aws_iam_role.ecs_task_execution_role.arn
  task_role_arn      = aws_iam_role.ecs_task_role.arn

  container_definitions = <<DEFINITION
[
  {
    "cpu": ${var.controlplane_httpserver_task_cpu},
    "image": "${element(split(":", var.controlplane_httpserver_image), 0)}:${data.external.td_httpserver.result["image_tag"]}",
    "memory": ${var.controlplane_httpserver_task_memory},
    "name": "${var.controlplane_httpserver_app_name}",
    "environment": [
      {
        "name": "ENVIRONMENT",
        "value": "${var.environment}"
      },
      {
        "name": "AWS_REGION",
        "value": "${var.region}"
      },
      {
        "name": "SANDSTORM_ROLE_ARN",
        "value": "${local.controlplane_sandstorm_roles[var.environment]}"
      },
      {
        "name": "POSTGRES_HOST",
        "value": "${var.rds_writer_hostname}"
      },
      {
        "name": "POSTGRES_WRITER_HOST",
        "value": "${var.rds_writer_hostname}"
      },
      {
        "name": "POSTGRES_READER_HOST",
        "value": "${var.rds_reader_hostname}"
      },
      {
        "name": "SLACK_CHANNEL",
        "value": "${var.slack_channel}"
      },
      {
        "name": "CW_ENABLED",
        "value": "true"
      },
      {
        "name": "AUTOPROF_ENABLED",
        "value": "true"
      },
      {
        "name": "AUTOPROF_BUCKET_NAME",
        "value": "${module.autoprof_controlplane_httpserver.autoprof_bucket_name}"
      },
      {
        "name": "AUTHORIZED_FIELD_KEY_ARN",
        "value": "${var.authorized_field_key_arn}"
      },
      {
        "name": "ENCRYPTION_AT_REST_KEY_ARN",
        "value": "${var.encryption_at_rest_key_arn}"
      }
    ],
    "essential": true,
    "logConfiguration": {
      "logDriver": "awslogs",
      "options": {
        "awslogs-group": "${aws_cloudwatch_log_group.controlplane_httpserver_logs.name}",
        "awslogs-region": "${var.region}",
        "awslogs-stream-prefix": "${var.cluster_name}"
      }
    },
    "placementConstraints": [
      {
        "type": "distinctInstance"
      }
    ],
    "portMappings": [
      {
        "containerPort": ${var.controlplane_httpserver_app_port},
        "protocol": "tcp"
      }
    ],
    "dnsServers": ["172.17.0.1"],
    "ulimits": [{
      "name": "nofile",
      "softLimit": 65535,
      "hardLimit": 65535
    }]
  }
]
DEFINITION

}

resource "aws_ecs_service" "controlplane_httpserver" {
  name            = var.controlplane_httpserver_app_name
  cluster         = aws_ecs_cluster.main.id
  task_definition = "${aws_ecs_task_definition.controlplane_httpserver.family}:${data.external.td_httpserver.result["task_definition_revision"] > aws_ecs_task_definition.controlplane_httpserver.revision ? data.external.td_httpserver.result["task_definition_revision"] : aws_ecs_task_definition.controlplane_httpserver.revision}"
  desired_count   = var.controlplane_httpserver_app_count
  iam_role        = var.service_iam_role

  load_balancer {
    target_group_arn = coalesce(
      var.target_group_id,
      aws_alb_target_group.controlplane_httpserver.id,
    )
    container_name = var.controlplane_httpserver_app_name
    container_port = var.controlplane_httpserver_app_port
  }

  depends_on = [
    aws_alb_listener.controlplane_httpserver_http,
    aws_alb_listener.controlplane_httpserver_https,
  ]
}

resource "aws_ecs_task_definition" "controlplane_converger" {
  family             = var.controlplane_converger_app_name
  network_mode       = "bridge"
  execution_role_arn = aws_iam_role.ecs_task_execution_role.arn
  task_role_arn      = aws_iam_role.ecs_task_role.arn

  container_definitions = <<DEFINITION
[
  {
    "cpu": ${var.controlplane_converger_task_cpu},
    "image": "${element(split(":", var.controlplane_converger_image), 0)}:${data.external.td_converger.result["image_tag"]}",
    "memory": ${var.controlplane_converger_task_memory},
    "name": "${var.controlplane_converger_app_name}",
    "environment": [
      {
        "name": "ENVIRONMENT",
        "value": "${var.environment}"
      },
      {
        "name": "AWS_REGION",
        "value": "${var.region}"
      },
      {
        "name": "SANDSTORM_ROLE_ARN",
        "value": "${local.controlplane_sandstorm_roles[var.environment]}"
      },
      {
        "name": "POSTGRES_HOST",
        "value": "${var.rds_writer_hostname}"
      },
      {
        "name": "POSTGRES_WRITER_HOST",
        "value": "${var.rds_writer_hostname}"
      },
      {
        "name": "POSTGRES_READER_HOST",
        "value": "${var.rds_reader_hostname}"
      },
      {
        "name": "SLACK_CHANNEL",
        "value": "${var.slack_channel}"
      },
      {
        "name": "CW_ENABLED",
        "value": "true"
      },
      {
        "name": "AUTOPROF_ENABLED",
        "value": "true"
      },
      {
        "name": "AUTOPROF_BUCKET_NAME",
        "value": "${module.autoprof_controlplane_converger.autoprof_bucket_name}"
      },
      {
        "name": "AUTHORIZED_FIELD_KEY_ARN",
        "value": "${var.authorized_field_key_arn}"
      },
      {
        "name": "ENCRYPTION_AT_REST_KEY_ARN",
        "value": "${var.encryption_at_rest_key_arn}"
      }
    ],
    "essential": true,
    "logConfiguration": {
      "logDriver": "awslogs",
      "options": {
        "awslogs-group": "${aws_cloudwatch_log_group.controlplane_converger_logs.name}",
        "awslogs-region": "${var.region}",
        "awslogs-stream-prefix": "${var.cluster_name}"
      }
    },
    "placementConstraints": [
      {
        "type": "distinctInstance"
      }
    ],
    "portMappings": [
      {
        "containerPort": ${var.controlplane_converger_app_port},
        "protocol": "tcp"
      }
    ],
    "dnsServers": ["172.17.0.1"],
    "ulimits": [{
      "name": "nofile",
      "softLimit": 65535,
      "hardLimit": 65535
    }]
  }
]
DEFINITION

}

resource "aws_ecs_service" "controlplane_converger" {
  name            = var.controlplane_converger_app_name
  cluster         = aws_ecs_cluster.main.id
  task_definition = "${aws_ecs_task_definition.controlplane_converger.family}:${data.external.td_converger.result["task_definition_revision"] > aws_ecs_task_definition.controlplane_converger.revision ? data.external.td_converger.result["task_definition_revision"] : aws_ecs_task_definition.controlplane_converger.revision}"
  desired_count   = var.controlplane_converger_app_count
}

locals {
  controlplane_sandstorm_roles = {
    prod    = "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/eventbus-controlplane-production"
    staging = "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/eventbus-controlplane-staging"
  }
}

resource "aws_iam_policy" "controlplane_sandstorm_policy" {
  name = "controlplane_sandstorm_policy"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Resource": [
        "${local.controlplane_sandstorm_roles[var.environment]}"
      ],
      "Effect": "Allow"
    }
  ]
}
EOF

}

resource "aws_iam_policy" "controlplane_kms_policy" {
  name = "controlplane_kms_policy"

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": {
        "Effect": "Allow",
        "Action": [
            "kms:DescribeKey",
            "kms:GetKeyPolicy",
            "kms:PutKeyPolicy",
            "kms:CreateGrant",
            "kms:RevokeGrant",
            "kms:ListGrants"
        ],
        "Resource": "*"
    }
}
EOF

}

resource "aws_iam_role_policy_attachment" "controlplane_sandstorm_attach" {
  role       = aws_iam_role.ecs_task_role.name
  policy_arn = aws_iam_policy.controlplane_sandstorm_policy.arn
}

resource "aws_iam_role_policy_attachment" "controlplane_kms_attach" {
  role       = aws_iam_role.ecs_task_role.name
  policy_arn = aws_iam_policy.controlplane_kms_policy.arn
}

data "external" "td_httpserver" {
  program = ["bash", "${path.root}/../../scripts/ecs-task-definition.sh"]

  query = {
    service = var.controlplane_httpserver_app_name
    cluster = aws_ecs_cluster.main.id
  }
}

data "external" "td_converger" {
  program = ["bash", "${path.root}/../../scripts/ecs-task-definition.sh"]

  query = {
    service = var.controlplane_converger_app_name
    cluster = aws_ecs_cluster.main.id
  }
}

