resource "aws_alb" "dashboard_alb" {
  name            = var.dashboard_app_name
  internal        = true
  subnets         = module.twitch_networking.private_subnets
  security_groups = [module.twitch_networking.twitch_subnets_sg_id]
}

resource "aws_alb_target_group" "dashboard_tg" {
  deregistration_delay = 5
  slow_start           = var.lb_slow_start_seconds
  name                 = var.dashboard_app_name
  protocol             = "HTTP"
  vpc_id               = module.twitch_networking.vpc_id

  # The port is required, but overriden by ECS. Using a placeholder here.
  port = "1"

  health_check {
    protocol            = "HTTP"
    path                = "/health"
    healthy_threshold   = 3
    unhealthy_threshold = 2
    timeout             = 2
    interval            = 5
    matcher             = 200
  }
}

resource "aws_alb_listener" "dashboard_http" {
  load_balancer_arn = aws_alb.dashboard_alb.id
  port              = "80"
  protocol          = "HTTP"

  default_action {
    target_group_arn = aws_alb_target_group.dashboard_tg.id
    type             = "forward"
  }
}

resource "aws_alb_listener" "dashboard_https" {
  load_balancer_arn = aws_alb.dashboard_alb.id
  port              = "443"
  protocol          = "HTTPS"
  certificate_arn   = aws_acm_certificate.dashboard_ssl_cert_a2z.arn

  default_action {
    target_group_arn = aws_alb_target_group.dashboard_tg.id
    type             = "forward"
  }
}

# Deprecated certificate, used as optional during migration
# TODO: remove after cutover is complete
resource "aws_lb_listener_certificate" "deprecated_cert" {
  listener_arn    = "${aws_alb_listener.dashboard_https.arn}"
  certificate_arn = "${aws_acm_certificate.dashboard_ssl_cert.arn}"
}

resource "aws_acm_certificate" "dashboard_ssl_cert" {
  domain_name               = "eventbus-dashboard-${var.environment}.internal.justin.tv"
  validation_method         = "DNS"
  subject_alternative_names = var.subject_alternative_name

  tags = {
    app = var.dashboard_app_name
    env = var.environment
  }

  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_cloudwatch_log_group" "dashboard-logs" {
  name              = "${var.dashboard_app_name}-logs"
  retention_in_days = var.app_log_retention_days

  tags = {
    app = var.dashboard_app_name
    env = var.environment
  }
}

resource "aws_ecs_task_definition" "dashboard" {
  family             = var.dashboard_app_name
  network_mode       = "bridge"
  execution_role_arn = aws_iam_role.ecs_task_execution_role.arn
  task_role_arn      = aws_iam_role.ecs_task_role.arn

  container_definitions = <<DEFINITION
[
  {
    "cpu": ${var.dashboard_task_cpu},
    "image": "${element(split(":", var.dashboard_image), 0)}:${data.external.td_dashboard.result["image_tag"]}",
    "memory": ${var.dashboard_task_memory},
    "name": "${var.dashboard_app_name}",
    "environment": [
      {
        "name": "ENVIRONMENT",
        "value": "${var.environment}"
      }
    ],
    "essential": true,
    "logConfiguration": {
      "logDriver": "awslogs",
      "options": {
        "awslogs-group": "${aws_cloudwatch_log_group.dashboard-logs.name}",
        "awslogs-region": "${var.region}",
        "awslogs-stream-prefix": "${var.dashboard_app_name}"
      }
    },
    "placementConstraints": [
      {
        "type": "distinctInstance"
      }
    ],
    "portMappings": [
      {
        "containerPort": ${var.dashboard_app_port},
        "protocol": "tcp"
      }
    ],
    "dnsServers": ["172.17.0.1"],
    "ulimits": [{
      "name": "nofile",
      "softLimit": 65535,
      "hardLimit": 65535
    }]
  }
]
DEFINITION

}

resource "aws_ecs_service" "dashboard" {
  name            = var.dashboard_app_name
  cluster         = aws_ecs_cluster.main.id
  task_definition = "${aws_ecs_task_definition.dashboard.family}:${data.external.td_dashboard.result["task_definition_revision"] > aws_ecs_task_definition.dashboard.revision ? data.external.td_dashboard.result["task_definition_revision"] : aws_ecs_task_definition.dashboard.revision}"
  desired_count   = var.dashboard_app_count
  iam_role        = var.service_iam_role

  load_balancer {
    target_group_arn = coalesce(
      aws_alb_target_group.dashboard_tg.id,
      aws_alb_target_group.dashboard_tg.id,
    )
    container_name = var.dashboard_app_name
    container_port = var.dashboard_app_port
  }

  depends_on = [
    aws_alb_listener.dashboard_http,
    aws_alb_listener.dashboard_https,
  ]
}

# Sandstorm dashboard roles

locals {
  dashboard_sandstorm_roles = {
    prod    = "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/eventbus-dashboard-production"
    staging = "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/eventbus-dashboard-staging"
  }
}

resource "aws_iam_policy" "dashboard_sandstorm_policy" {
  name = "dashboard_sandstorm_policy"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Resource": [
        "${local.dashboard_sandstorm_roles[var.environment]}"
      ],
      "Effect": "Allow"
    }
  ]
}
EOF

}

resource "aws_iam_role_policy_attachment" "dashboard_sandstorm_attach" {
  role       = aws_iam_role.ecs_task_role.name
  policy_arn = aws_iam_policy.dashboard_sandstorm_policy.arn
}

data "external" "td_dashboard" {
  program = ["bash", "${path.root}/../../scripts/ecs-task-definition.sh"]

  query = {
    service = var.dashboard_app_name
    cluster = aws_ecs_cluster.main.id
  }
}

