module "twitch_networking" {
  source = "../twitch_networking"

  environment = var.environment
}

data "aws_iam_account_alias" "current" {
}

data "aws_caller_identity" "current" {
}

data "aws_iam_policy" "s2s_assume_role_policy" {
  arn = "arn:aws:iam::${data.aws_caller_identity.current.account_id}:policy/S2SAssumeRole"
}

locals {

  acct_alias = data.aws_iam_account_alias.current.account_alias
  ldap_host  = "ldap.twitch.a2z.com"
}

resource "aws_iam_role_policy_attachment" "allow_send_access_logs_policy" {
  role       = var.ecs_host_iam_id
  policy_arn = "arn:aws:iam::aws:policy/AWSOpsWorksCloudWatchLogs"
}

resource "aws_cloudwatch_log_group" "ssm_logs" {
  name              = "${var.cluster_name}-ssm-ec2-logs"
  retention_in_days = var.app_log_retention_days

  tags = {
    app     = var.controlplane_httpserver_app_name
    cluster = var.cluster_name
    env     = var.environment
  }
}

resource "aws_cloudwatch_log_group" "ecs_logs" {
  name              = "${var.cluster_name}-ecs-logs"
  retention_in_days = var.app_log_retention_days

  tags = {
    cluster = var.cluster_name
    env     = var.environment
  }
}

resource "aws_ecs_cluster" "main" {
  name = var.cluster_name
}

resource "aws_iam_role" "ecs_task_execution_role" {
  name = "ecs_task_execution_${var.environment}"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "ecs-tasks.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF

}

resource "aws_iam_role_policy_attachment" "ecs_task_execution_role" {
  role       = aws_iam_role.ecs_task_execution_role.id
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonECSTaskExecutionRolePolicy"
}

resource "aws_iam_role" "ecs_task_role" {
  name = "ecs_task_role_${var.environment}"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "ecs-tasks.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF

}

resource "aws_iam_role_policy_attachment" "s3_role" {
  role       = aws_iam_role.ecs_task_role.id
  policy_arn = "arn:aws:iam::aws:policy/AmazonS3FullAccess"
}

resource "aws_iam_role_policy_attachment" "s2s_assume_role_attachment" {
  role       = aws_iam_role.ecs_task_role.id
  policy_arn = data.aws_iam_policy.s2s_assume_role_policy.arn
}

# Setup autoprof for each service
module "autoprof_controlplane_httpserver" {
  source       = "../../modules/autoprof"
  service_name = "controlplane-httpserver"
  environment  = var.environment
}

resource "aws_iam_role_policy_attachment" "autoprof_controlplane_httpserver_attachment" {
  role       = aws_iam_role.ecs_task_role.id
  policy_arn = module.autoprof_controlplane_httpserver.autoprof_iam_policy_arn
}

module "autoprof_controlplane_converger" {
  source       = "../../modules/autoprof"
  service_name = "controlplane-converger"
  environment  = var.environment
}

resource "aws_iam_role_policy_attachment" "autoprof_controlplane_converger_attachment" {
  role       = aws_iam_role.ecs_task_role.id
  policy_arn = module.autoprof_controlplane_converger.autoprof_iam_policy_arn
}

resource "aws_iam_policy" "cloudwatch_write_access" {
  name        = "EventBusCloudWatch"
  path        = "/"
  description = "Allow eventbus applications to publish CloudWatch metrics"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "cloudwatch:PutMetricData"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
EOF

}

resource "aws_iam_role_policy_attachment" "cloudwatch_policy_attachment" {
  role       = aws_iam_role.ecs_task_role.id
  policy_arn = aws_iam_policy.cloudwatch_write_access.arn
}

resource "aws_iam_policy" "cloudwatch_read_access" {
  name        = "EventBusCloudWatchReadAccess"
  path        = "/"
  description = "Allow eventbus controlplane to fetch event statistics"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "cloudwatch:GetMetricData"
      ],
      "Effect": "Allow",
      "Resource": "*"
    }
  ]
}
EOF

}

resource "aws_iam_role_policy_attachment" "cloudwatch_read_access_policy_attachment" {
  role       = aws_iam_role.ecs_task_role.id
  policy_arn = aws_iam_policy.cloudwatch_read_access.arn
}

# Define how our upcoming ECS cluster will get autoscaled & how the cluster's EC2's are deployed and configured
# Note: installs LDAP-AUTH on the ec2's pointing to our special LDAP hosts
module "ecs_asg" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//ecsasg?ref=v2.1.0"

  cluster_name                  = var.cluster_name
  task_cleanup_wait_duration    = var.task_cleanup_wait_duration
  container_instance_profile_id = var.ecs_host_iam_id
  desired_capacity              = var.cluster_size["desired"]
  max_size                      = var.cluster_size["max"]
  min_size                      = var.cluster_size["min"]
  environment                   = var.environment
  instance_type                 = var.ec2_instance_type
  ldap_group                    = var.ssh_sudoer_ldap_group
  custom_ldap_host              = local.ldap_host
  name                          = var.cluster_name
  owner                         = var.owner_email

  ecs_init_loggroup  = aws_cloudwatch_log_group.ecs_logs.name
  ecs_agent_loggroup = aws_cloudwatch_log_group.ecs_logs.name

  ssm_agent_loggroup = aws_cloudwatch_log_group.ssm_logs.name
  team               = var.team_name

  security_groups = module.twitch_networking.twitch_subnets_sg_id

  # Subnets to which ASG will add new ec2's
  vpc_zone_identifier = join(",", module.twitch_networking.private_subnets)

  # How do we scale up in case of high cpu?
  scale_out_cooldown           = 1200
  scale_out_evaluation_periods = 5
  scale_out_increment          = 1

  # set really high - let's avoid unnecessary scaling up
  scale_out_cpu_percent_threshold = 90

  # How do we scale down in case of low cpu?
  scale_in_cooldown           = 900
  scale_in_evaluation_periods = 15
  scale_in_increment          = -1

  # set really low - let's avoid unnecessary scaling down
  scale_in_cpu_percent_threshold = 0.01
}

resource "aws_iam_service_linked_role" "ecs" {
  aws_service_name = "ecs.amazonaws.com"
}

resource "aws_iam_role_policy" "sns_policy" {
  name   = "sns_policy"
  policy = data.aws_iam_policy_document.sns_policy.json
  role   = aws_iam_role.ecs_task_role.name
}

data "aws_iam_policy_document" "sns_policy" {
  statement {
    actions   = ["sns:*"]
    effect    = "Allow"
    resources = ["*"]
  }
}

# Allows the controlplane's ECS task to assume to EventBus roles in other accounts.
# Those accounts still need to permission us to do that assumption, this just makes
# the task role able to do so.

resource "aws_iam_role_policy" "eventbus_assume_policy" {
  name   = "eventbus_assume_policy"
  policy = data.aws_iam_policy_document.eventbus_assume_policy.json
  role   = aws_iam_role.ecs_task_role.name
}

data "aws_iam_policy_document" "eventbus_assume_policy" {
  statement {
    actions   = ["sts:AssumeRole"]
    effect    = "Allow"
    resources = ["arn:aws:iam::*:role/EventBus*"]
  }
}

