
resource "aws_cloudwatch_log_group" "controlplane_validator_logs" {
  name              = "${var.controlplane_validator_app_name}-logs"
  retention_in_days = var.app_log_retention_days

  tags = {
    app = var.controlplane_validator_app_name
    env = var.environment
  }
}

module "autoprof_controlplane_validator" {
  source       = "../../modules/autoprof"
  service_name = "controlplane-validator"
  environment  = var.environment
}

resource "aws_iam_role_policy_attachment" "autoprof_controlplane_validator_attachment" {
  role       = aws_iam_role.ecs_task_role.id
  policy_arn = module.autoprof_controlplane_validator.autoprof_iam_policy_arn
}

resource "aws_ecs_task_definition" "controlplane_validator" {
  family             = var.controlplane_validator_app_name
  network_mode       = "bridge"
  execution_role_arn = aws_iam_role.ecs_task_execution_role.arn
  task_role_arn      = aws_iam_role.ecs_task_role.arn

  container_definitions = <<DEFINITION
[
  {
    "cpu": ${var.controlplane_validator_task_cpu},
    "image": "${element(split(":", var.controlplane_validator_image), 0)}:${data.external.td_validator.result["image_tag"]}",
    "memory": ${var.controlplane_validator_task_memory},
    "name": "${var.controlplane_validator_app_name}",
    "environment": [
      {
        "name": "ENVIRONMENT",
        "value": "${var.environment}"
      },
      {
        "name": "AWS_REGION",
        "value": "${var.region}"
      },
      {
        "name": "SANDSTORM_ROLE_ARN",
        "value": "${local.controlplane_sandstorm_roles[var.environment]}"
      },
      {
        "name": "POSTGRES_HOST",
        "value": "${var.rds_writer_hostname}"
      },
      {
        "name": "POSTGRES_WRITER_HOST",
        "value": "${var.rds_writer_hostname}"
      },
      {
        "name": "POSTGRES_READER_HOST",
        "value": "${var.rds_reader_hostname}"
      },
      {
        "name": "SLACK_CHANNEL",
        "value": "${var.controlplane_validator_slack_channel}"
      },
      {
        "name": "CW_ENABLED",
        "value": "true"
      },
      {
        "name": "AUTOPROF_ENABLED",
        "value": "true"
      },
      {
        "name": "AUTOPROF_BUCKET_NAME",
        "value": "${module.autoprof_controlplane_validator.autoprof_bucket_name}"
      },
      {
        "name": "AUTHORIZED_FIELD_KEY_ARN",
        "value": "${var.authorized_field_key_arn}"
      },
      {
        "name": "ENCRYPTION_AT_REST_KEY_ARN",
        "value": "${var.encryption_at_rest_key_arn}"
      },
      {
        "name": "PAGERDUTY_ENABLED",
        "value": "${var.controlplane_validator_pagerduty_enabled}"
      },
      {
        "name": "PAGERDUTY_HIGH_URGENCY_KEY",
        "value": "${var.controlplane_validator_pagerduty_high_urgency_key}"
      },
      {
        "name": "PAGERDUTY_LOW_URGENCY_KEY",
        "value": "${var.controlplane_validator_pagerduty_low_urgency_key}"
      }
    ],
    "essential": true,
    "logConfiguration": {
      "logDriver": "awslogs",
      "options": {
        "awslogs-group": "${aws_cloudwatch_log_group.controlplane_validator_logs.name}",
        "awslogs-region": "${var.region}",
        "awslogs-stream-prefix": "${var.cluster_name}"
      }
    },
    "placementConstraints": [
      {
        "type": "distinctInstance"
      }
    ],
    "portMappings": [
      {
        "containerPort": ${var.controlplane_validator_app_port},
        "protocol": "tcp"
      }
    ],
    "dnsServers": ["172.17.0.1"],
    "ulimits": [{
      "name": "nofile",
      "softLimit": 65535,
      "hardLimit": 65535
    }]
  }
]
DEFINITION

}

resource "aws_ecs_service" "controlplane_validator" {
  name            = var.controlplane_validator_app_name
  cluster         = aws_ecs_cluster.main.id
  task_definition = "${aws_ecs_task_definition.controlplane_validator.family}:${data.external.td_validator.result["task_definition_revision"] > aws_ecs_task_definition.controlplane_validator.revision ? data.external.td_validator.result["task_definition_revision"] : aws_ecs_task_definition.controlplane_validator.revision}"
  desired_count   = var.controlplane_validator_app_count
}

data "external" "td_validator" {
  program = ["bash", "${path.root}/../../scripts/ecs-task-definition.sh"]

  query = {
    service = var.controlplane_validator_app_name
    cluster = aws_ecs_cluster.main.id
  }
}
