# Create a policy that allows Lambda to start logging and send SNS alerts
resource "aws_iam_policy" "cloudtrail_lambda_policy" {
  name        = "lambda_cloudtrail_alert_policy"
  path        = "/"
  description = "Enables StartLogging and SNS Notifications for Lambda"

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": [
                "logs:CreateLogGroup",
                "logs:CreateLogStream",
                "logs:PutLogEvents"
                ],        
            "Resource": "arn:aws:logs:*:*:*"
        },
        {
            "Effect": "Allow",
            "Action": [
                "sns:Publish"
            ],
            "Resource": "arn:aws:sns:us-west-2:508423404353:twitch-sirt"
        },
        {
            "Effect": "Allow",
            "Action": [
                "cloudtrail:StartLogging"
            ],
            "Resource": [
                "arn:aws:cloudtrail:*"
            ]
        }
    ]
}
EOF

}

# Create a role that allows lambda execution 

resource "aws_iam_role" "cloudtrail_lambda_role" {
  name = "lambda_execute_cloudtrail_role"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "",
      "Effect": "Allow",
      "Principal": {
        "Service": "lambda.amazonaws.com"
      },
      "Action": "sts:AssumeRole"
    }
  ]
}
EOF

}

# Attach lambda policy to role
resource "aws_iam_role_policy_attachment" "cloudtrail_lambda_policy_attach" {
  role       = aws_iam_role.cloudtrail_lambda_role.name
  policy_arn = aws_iam_policy.cloudtrail_lambda_policy.arn
}

# Create the lambda
resource "aws_lambda_function" "cloudtrail_lambda_function" {
  filename         = "${path.module}/lambda.zip"
  function_name    = "lambda_cloudtrail_monitor_SECURITY"
  role             = aws_iam_role.cloudtrail_lambda_role.arn
  handler          = "lambda_cloudtrail_alert.lambda_handler"
  source_code_hash = filebase64sha256("${path.module}/lambda.zip")
  runtime          = "python2.7"

  lifecycle {
    ignore_changes = [filename]
  }
}

# Create a rule to fire the lambda
resource "aws_cloudwatch_event_rule" "console" {
  name        = "cloudwatch-event-guard"
  description = "Triggers on StopLogging, DeleteTrail, and UpdateTrail"

  event_pattern = <<PATTERN
{
  "detail":
    {
     "eventName": [ "StopLogging","DeleteTrail","UpdateTrail"],
     "eventSource":["cloudtrail.amazonaws.com"]
    },
  "detail-type":["AWS API Call via CloudTrail"],
  "source":["aws.cloudtrail"]
}
PATTERN

}

# define the target for the rule

resource "aws_cloudwatch_event_target" "lambda_target" {
  rule      = aws_cloudwatch_event_rule.console.name
  target_id = "1-cloudwatch-event-guard"
  arn       = aws_lambda_function.cloudtrail_lambda_function.arn
}

# Add lambda invoke permission to rule
resource "aws_lambda_permission" "allow_cloudwatch" {
  statement_id  = "lambda_cloudtrail_monitor_${data.aws_caller_identity.current.account_id}"
  action        = "lambda:InvokeFunction"
  function_name = aws_lambda_function.cloudtrail_lambda_function.function_name
  principal     = "events.amazonaws.com"
  source_arn    = "arn:aws:events:${data.aws_region.current.name}:${data.aws_caller_identity.current.account_id}:rule/cloudwatch-event-guard"
}

