locals {
  db_backup_retention_days    = 30
  db_instance_count           = 3
  db_max_cpu_util             = 50
  db_engine                   = "aurora-postgresql"
  db_engine_version           = "10.14"
  monitoring_interval_seconds = 10
}

module "twitch_networking" {
  source      = "../twitch_networking"
  environment = var.environment
}

data "aws_iam_policy_document" "db_monitoring_assume" {
  statement {
    actions = ["sts:AssumeRole"]
    effect  = "Allow"

    principals {
      identifiers = ["monitoring.rds.amazonaws.com"]
      type        = "Service"
    }
  }
}

resource "aws_iam_role" "db_monitoring" {
  assume_role_policy = data.aws_iam_policy_document.db_monitoring_assume.json
  name               = "eventbus-rds-monitoring"
}

resource "aws_iam_role_policy_attachment" "db_monitoring" {
  role       = aws_iam_role.db_monitoring.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonRDSEnhancedMonitoringRole"
}

resource "aws_db_parameter_group" "eventbus" {
  name        = "eventbus-${var.environment}"
  family      = "aurora-postgresql10"
  description = "EventBus parameter group"

  parameter {
    name  = "log_min_duration_statement"
    value = "2000"
  }
}

resource "aws_db_subnet_group" "db_subnet_group" {
  description = "EventBus subnet group"
  name        = "eventbus-${var.environment}"
  subnet_ids  = module.twitch_networking.private_subnets
}

resource "aws_rds_cluster" "db_cluster" {
  backup_retention_period = local.db_backup_retention_days
  cluster_identifier      = "eventbus-${var.environment}"
  database_name           = "eventbus"
  db_subnet_group_name    = aws_db_subnet_group.db_subnet_group.name
  engine                  = local.db_engine
  engine_version          = local.db_engine_version
  master_username         = "twitch"
  master_password         = "changeme"

  # 2:00-3:00am PDT
  preferred_backup_window = "09:00-10:00"

  # Tuesday 1:00-2:00pm PDT
  preferred_maintenance_window = "tue:20:00-tue:21:00"
  storage_encrypted            = true
  vpc_security_group_ids       = [module.twitch_networking.twitch_subnets_sg_id]

  lifecycle {
    ignore_changes = [master_password]
  }
}

resource "aws_rds_cluster_instance" "db_instance" {
  count = local.db_instance_count

  cluster_identifier      = aws_rds_cluster.db_cluster.cluster_identifier
  db_parameter_group_name = aws_db_parameter_group.eventbus.name
  db_subnet_group_name    = aws_db_subnet_group.db_subnet_group.name
  engine                  = local.db_engine
  identifier              = "eventbus-${var.environment}-${count.index}"
  instance_class          = "db.r5.large"
  monitoring_interval     = local.monitoring_interval_seconds
  monitoring_role_arn     = aws_iam_role.db_monitoring.arn
}

