#!bin/python

# Lists status of read/write capacity alarms for all dynamo tables in account

from termcolor import colored
import boto3
import json
import decimal
import requests
import time
from boto3.dynamodb.conditions import Key, Attr

# Helper class to convert a DynamoDB item to JSON.
class DecimalEncoder(json.JSONEncoder):
    def default(self, o):
        if isinstance(o, decimal.Decimal):
            if o % 1 > 0:
                return float(o)
            else:
                return int(o)
        return super(DecimalEncoder, self).default(o)

session = boto3.Session(region_name='us-west-2')
dynamoClient = session.client("dynamodb")

table_list = dynamoClient.list_tables()

tables = {}
tableNameMaxLength = 0
GSIMaxLength = 0

for tableName in table_list["TableNames"]:
    tables[tableName] = { "ReadAlert": colored(u'\u2718', 'red'), "WriteAlert": colored(u'\u2718', 'red'), "indexes": {} }

    if len(tableName) > tableNameMaxLength:
        tableNameMaxLength = len(tableName)

    description = dynamoClient.describe_table(TableName = tableName)

    if "GlobalSecondaryIndexes" in description["Table"]:
        for GSI in description["Table"]["GlobalSecondaryIndexes"]:
            tables[tableName]["indexes"][GSI["IndexName"]] = { "ReadAlert": colored(u'\u2718', 'red'), "WriteAlert": colored(u'\u2718', 'red') }
            if len(GSI["IndexName"]) > GSIMaxLength:
                GSIMaxLength = len(GSI["IndexName"])

# print(json.dumps(tables, indent=4, sort_keys=True, default=str))

cloudwatch = session.client("cloudwatch")

response = cloudwatch.describe_alarms()

while True:

    for alarm in response["MetricAlarms"]:
        if alarm["Namespace"] != "AWS/DynamoDB":
            continue
    
        #print(json.dumps(alarm, indent=4, sort_keys=True, default=str))

        tableName = None
        GSI = None
        type = alarm["MetricName"]
        action = alarm["AlarmActions"]
    
        for dimension in alarm["Dimensions"]:
            if dimension["Name"] == "TableName":
                tableName = dimension["Value"]
            elif dimension["Name"] == "GlobalSecondaryIndexName":
                GSI = dimension["Value"]
            else:
                print("unknown dimension name %s.", dimension["Name"])
    
        foo = { "table": tableName, "index": GSI, "type": type, "action": action }
        #print(json.dumps(foo, indent=4, sort_keys=True, default=str))
    
        if tableName not in tables:
            continue

        if GSI == None:
            if type == "ConsumedReadCapacityUnits":
                tables[tableName]["ReadAlert"] = colored(u'\u2714', 'green')
            elif type == "ConsumedWriteCapacityUnits":
                tables[tableName]["WriteAlert"] = colored(u'\u2714', 'green')
        else:
            if type == "ConsumedReadCapacityUnits":
                tables[tableName]["indexes"][GSI]["ReadAlert"] = colored(u'\u2714', 'green')
            elif type == "ConsumedWriteCapacityUnits":
                tables[tableName]["indexes"][GSI]["WriteAlert"] = colored(u'\u2714', 'green')

    if "NextToken" not in response:
        break

    response = cloudwatch.describe_alarms(NextToken = response["NextToken"])

# print(json.dumps(tables, indent=4, sort_keys=True, default=str))

headerFormat = "%" + str(tableNameMaxLength) + "s\t%" + str(GSIMaxLength) + "s\t%12s\t%12s"
format = "%" + str(tableNameMaxLength) + "s\t%" + str(GSIMaxLength) + "s\t\t%12s\t\t%12s"

print(headerFormat % ("Table Name", "GSI Name (Optional)", "ReadAlert", "WriteAlert"))

for table, data in tables.items():
    print(format % (table, "", data["ReadAlert"], data["WriteAlert"]))

    for indexName, data in data["indexes"].items():
        print(format % (table, indexName, data["ReadAlert"], data["WriteAlert"]))
