#!/bin/bash
set -o pipefail

# this script will do a full docker->deb->publish flow for a given project.

BIN=$(dirname $BASH_SOURCE)
export PWD=$(pwd)

# Later modules will `set -u`, so this variable needs to be set now
DEFAULT_SERVICE_TEMPLATE=svc/service.yaml
if [ -z ${SERVICE_TEMPLATE} ]; then
	SERVICE_TEMPLATE=${DEFAULT_SERVICE_TEMPLATE}
fi
# LOAD all the modules
for f in `ls ${BIN}/modules/*.sh`; do
  source ${f}
done

# these come from the environment
#
# ORG - your organization at twitch, for example: gds
# PROJ - name of the project (folder in this repo) you're building. There are assumptions here, for exmaple
#        it is assumed you are using a Dockerfile-service for production builds, etc.
# GIT_COMMIT - the git commit that the docker image was built for, provided to spinnaker via spinnaker.properties


# ASSERT that the critical external variables are set
ORG=${ORG}
PROJ=${PROJ}
GIT_COMMIT=${GIT_COMMIT}

# CREATE build and output dirs
utils::prepdirs ${PWD}
  # exports: BUILD_DIR and OUTPUT_DIR

# BUILD docker container from the Dockerfile-service file in the PROJ folder
docker::build ${PROJ} ${ORG}
  # exports: DOCKER_TAG and DOCKER_VERSION (which is the twitch.version specified in the LABEL of the Dockerfile-service file)

# PUSH the docker container to the docker repo
docker::publish ${PROJ} ${ORG} ${DOCKER_TAG}
  # exports: DOCKER_SHA

# CREATE the service yaml file used by docker-compose
service::definition ${PROJ} ${ORG} ${DOCKER_SHA} ${BUILD_DIR} ${SERVICE_TEMPLATE}

# CREATE debain control file for debian build that follows
service::controlfile ${PROJ} ${ORG} ${DOCKER_VERSION} ${DOCKER_SHA} ${BUILD_DIR} ${GIT_COMMIT}

# COPY docker-compose upstart file into place
service::upstart ${BUILD_DIR}

# COPY postinstallation script into place
service::postinst ${BUILD_DIR}

# CREATE debian artifact from the BUILD_DIR into the OUTPUT_DIR
aptly::build ${PROJ} ${ORG} ${DOCKER_VERSION} ${DOCKER_SHA} ${BUILD_DIR} ${OUTPUT_DIR}
  # exports: APTLY_ARTIFACT

# PUBLISH apt artifact if new
aptly::maybe_publish ${PROJ} ${ORG} ${DOCKER_VERSION} ${DOCKER_SHA} ${APTLY_ARTIFACT}

# If we get to the end there's a chance that aptly::maybe_publish returned 1 - but we should exit 0
exit 0
