#!/bin/bash
set -euxo pipefail

# This script will build and run a container in the project ${PROJ},
# defaulting to Dockerfile-binary. This container needs to drop it's
# result into a volume mounted to /build. Next, the script will run
# the packager container, defaulting to
# docker.internal.justin.tv/gds/debbuilder, which will output by
# default a package ${ORG}-${PROJ}_${VERSION}_${GIT_VERSION}, where
# PROJ has been mangled to replace / with - and VERSION is fetched
# from the project's twitch.version label in the docker container.


# This is the name of the dockerfile that we will build from, defaults
# to Dockerfile-binary. This container needs to have a volume, /build,
# where it drops everything to be packaged.
DOCKER_SERVICE=${DOCKER_SERVICE:-"Dockerfile-binary"}

# Override this for testing
ORG=${ORG:-"gds"}

# This is the container to use that builds the package. Expects 2
# volumes, /build for the binary tree to package and /out for the
# package destination.
PACKAGE_CONTAINER=${PACKAGE_CONTAINER:-"docker.internal.justin.tv/gds/debbuilder:latest"}

# Override this variable to not use the ${PROJ} variable
PACKAGE_NAME=${PACKAGE_NAME:-${PROJ}}
PACKAGE_NAME=${ORG}-${PACKAGE_NAME//[\/_]/-}

# Override this for setting tthe destination path. Must be an absolute
# path.
PACKAGE_PATH=${PACKAGE_PATH:-"/tmp/gds_package"}

# Set GIT_COMMIT to have a version other than unknown
GIT_COMMIT=${GIT_COMMIT:-"unknown"}

# Set to 1 to upload package
UPLOAD_PACKAGE=${UPLOAD_PACKAGE:-""}

BIN=$(dirname $BASH_SOURCE)
source ${BIN}/modules/utils.sh
source ${BIN}/modules/docker.sh
source ${BIN}/modules/aptly.sh

binary_path=`mktemp -d /tmp/gds_binary.XXXXXX`
package_path="${PACKAGE_PATH}/output"
mkdir -p ${package_path}

# Build the docker container for the binary, capture the version output
docker::build ${PROJ} ${ORG}

# run the docker container for the binary
docker run -t -v ${binary_path}:/build ${DOCKER_CONTAINER}:${DOCKER_TAG}

PACKAGE_VERSION=${DOCKER_VERSION}-${GIT_COMMIT}

# Pull the packager container
docker pull ${PACKAGE_CONTAINER}

# Run the packager container
docker run -t -v ${binary_path}:/build -v ${package_path}:/out -e PACKAGE_NAME=${PACKAGE_NAME} -e PACKAGE_VERSION=${PACKAGE_VERSION} $PACKAGE_CONTAINER

rm -rf ${binary_path}
package_file="${package_path}/${PACKAGE_NAME}_${PACKAGE_VERSION}_amd64.deb"

echo "Package built in ${package_file}"

if [ "$UPLOAD_PACKAGE" ]; then
	echo "Uploading package ${package_file}"
	aptly::publish ${package_file}
fi
	
