#!/bin/bash
set -o pipefail

BIN=$(dirname $BASH_SOURCE)
export PWD=$(pwd)

for f in `ls ${BIN}/modules/*.sh`; do
  source ${f}
done

set +u

if [ $# -ne 1 ]; then
	echo "Expected an argument: missing directory for where to find the Dockerfile for this lambda"
	exit 1
fi

###
### -- usage: 
###
###	AWS_ACCOUNT=twitch_ds_dev AWS_ACCESS_KEY=foo AWS_SECRET_KEY=foo bin/deploy_lambda.sh extensions/ems/lambda/src/ExtensionsDynamoToES
###
### -- expectations:
###
###	* output/function.zip: the zip file containing the lambda to be deployed to AWS.
###
### -- environment variables:
###
###	* AWS_ACCOUNT: self-explanatory, mandatory that it is correct
###	* AWS_ACCESS_KEY_ID: required when running in jenkins, so it can run aws cli commands
###	* AWS_SECRET_ACCESS_KEY: ibid
###
### -- arguments:
###
###	* DOCKERFILE_PATH: the directory where the Dockerfile is for building the lambda function
###		e.g. extensions/ems/lambda/foo/src/DynamoToES,
###		as well as the ${AWS_ACCOUNT}.config.json file describing the deployment characteristics
###
### -- variables:
###
###	* APP: e.g. extensions, taken from the leading part of the DOCKERFILE_PATH
###	* COMPONENT: the name of the lambda function, e.g. DynamoToES, taken from the trailing part
###		of DOCKERFILE_PATH
###	* GIT_COMMIT: populated in jenkins, set to a default otherwise
###

DOCKERFILE_PATH=${1%/}
APP=${DOCKERFILE_PATH%%/*}
COMPONENT=${DOCKERFILE_PATH##*/}
GIT_COMMIT=${GIT_COMMIT:-"yolo manual push, could be anything"}
AWS_DEFAULT_REGION=${AWS_REGION:-"us-west-2"}

# make it available to all subshells
export AWS_DEFAULT_REGION

utils::notnull "APP" ${APP}
utils::notnull "COMPONENT" ${COMPONENT}
utils::notnull "GIT_COMMIT" ${GIT_COMMIT}
utils::notnull "AWS_ACCOUNT" ${AWS_ACCOUNT}
utils::notnull "AWS_DEFAULT_REGION" ${AWS_DEFAULT_REGION}

# It's really there, right?  And it's really a zipfile, right?

utils::check "Making sure zipfile exists and passes zip tests" \
	[ -f output/function.zip ] && zip -T output/function.zip

CHECKSUM=$(openssl dgst -binary -sha256 output/function.zip | openssl base64)

CONFIG_FILE=${DOCKERFILE_PATH}/${AWS_ACCOUNT}.config.json

utils::check "Making sure a file exists in ${CONFIG_FILE}" \
	[ -f ${CONFIG_FILE} ]

FUNCTION_NAME="${APP}-${COMPONENT}"
AWS_IAM_ROLE=$(${JQ} -r 'select(.lambda_role_arn != null) | .lambda_role_arn' ${CONFIG_FILE})
DESCRIPTION="$(${JQ} -r 'select(.description != null) | .description' ${CONFIG_FILE})"
RUNTIME=$(${JQ} -r 'select(.runtime != null) | .runtime' ${CONFIG_FILE})
MEMORY_SIZE=$(${JQ} -r 'select(.memory != null) | .memory' ${CONFIG_FILE})
TIMEOUT=$(${JQ} -r 'select(.timeout != null) | .timeout' ${CONFIG_FILE})
HANDLER=$(${JQ} -r 'select(.handler != null) | .handler' ${CONFIG_FILE})
ENVIRONMENT=$(${JQ} -cr 'select(.environment != null) | {"Variables": .environment}' ${CONFIG_FILE})
VPC_CONFIG=$(${JQ} -cr 'select(.vpc_config != null) | .vpc_config' ${CONFIG_FILE})

utils::notnull "AWS_IAM_ROLE" ${AWS_IAM_ROLE}
utils::notnull "DESCRIPTION" "${DESCRIPTION}"
utils::notnull "RUNTIME" ${RUNTIME}
utils::notnull "MEMORY_SIZE" ${MEMORY_SIZE}
utils::notnull "TIMEOUT", ${TIMEOUT}
utils::notnull "HANDLER", ${HANDLER}

OPTIONAL_ARGS=""

if [ ! -z "${ENVIRONMENT}" ]; then
	OPTIONAL_ARGS="${OPTIONAL_ARGS} --environment ${ENVIRONMENT}"
fi

if [ ! -z "${VPC_CONFIG}" ]; then
	OPTIONAL_ARGS="${OPTIONAL_ARGS} --vpc-config ${VPC_CONFIG}"
fi

DESCRIPTION="${DESCRIPTION} (git hash: ${GIT_COMMIT})"

FOUND=$(aws lambda get-function --function-name "${FUNCTION_NAME}" 2>&1)

if [ $? -eq 0 ]; then
	export FOUND
	EXISTING_CHECKSUM=$(echo "${FOUND}" | ${JQ} -r '.Configuration.CodeSha256')

	if [ "${EXISTING_CHECKSUM}" = "${CHECKSUM}" ]; then
		echo "Code has not changed (checksums match), not pushing code"
		exit 0
	fi

	utils::check "Updating lambda function code" \
		aws lambda update-function-code \
			--function-name ${FUNCTION_NAME} \
			--zip-file fileb://output/function.zip

	utils::check "Updating lambda configuration" \
		aws lambda update-function-configuration \
			--function-name ${FUNCTION_NAME} \
			--role ${AWS_IAM_ROLE} \
			--runtime ${RUNTIME} \
			--timeout ${TIMEOUT} \
			--memory-size ${MEMORY_SIZE} \
			--handler ${HANDLER} \
			${OPTIONAL_ARGS} \
			--description "${DESCRIPTION}"

else
	if [[ "${FOUND}" =~ "ResourceNotFoundException" ]]; then
		utils::check "Creating lambda function ${FUNCTION_NAME}" \
			aws lambda create-function \
				--function-name "${FUNCTION_NAME}" \
				--role ${AWS_IAM_ROLE} \
				--runtime ${RUNTIME} \
				--timeout ${TIMEOUT} \
				--memory-size ${MEMORY_SIZE} \
				--description "${DESCRIPTION}" \
				--handler ${HANDLER} \
				${OPTIONAL_ARGS} \
				--zip-file fileb://output/function.zip
	else
		echo "Not sure what happened, I don't understand what 'aws lambda get-function' returned and I need human help: ${FOUND}"
		exit 1
	fi
fi
