#!/bin/bash
set -o pipefail

BIN=$(dirname $BASH_SOURCE)
export PWD=$(pwd)

for f in `ls ${BIN}/modules/*.sh`; do
  source ${f}
done

set +u

if [ $# -ne 1 ]; then
	echo "Expected an argument: missing directory for where to find the Dockerfile for this lambda"
	exit 1
fi

###
### -- variables:
###
###	* DOCKERFILE_PATH: the directory where the Dockerfile is for building the lambda function
###		e.g. extensions/ems/lambda/foo/src/DynamoToES
###	* APP: e.g. extensions, taken from the leading part of the DOCKERFILE_PATH
###	* COMPONENT: the name of the lambda function, e.g. DynamoToES, taken from the trailing part
###		of DOCKERFILE_PATH
###	* GIT_COMMIT: populated in jenkins, set to a default otherwise
###

DOCKERFILE_PATH=${1%/}
APP=${DOCKERFILE_PATH%%/*}
COMPONENT=${DOCKERFILE_PATH##*/}
GIT_COMMIT=${GIT_COMMIT:-"building_a_lambda_on_my_laptop"}

utils::notnull "APP" ${APP}
utils::notnull "COMPONENT" ${COMPONENT}

if [ -f ${DOCKERFILE_PATH}/.dockerignore ]; then
	echo "updating docker ignore from lambda-specific ignore file"
	cp ${DOCKERFILE_PATH}/.dockerignore .
else
	echo "updating docker ignore"
	utils::check "updating docker ignore" \
		bin/gen_dockerignore ${APP}
fi

echo "building docker image"
utils::check "building docker image" \
	docker build \
		--pull --no-cache \
		--build-arg APP=${APP} \
		--build-arg COMPONENT=${COMPONENT} \
		--build-arg PROJDIR=${DOCKERFILE_PATH} \
		-t ${GIT_COMMIT} \
		-f ${DOCKERFILE_PATH}/Dockerfile .

mkdir -p output
rm -rf output/*

echo "copying function.zip out of docker image"
utils::check "copying function.zip out of docker image" \
	echo "cp /function.zip /output/function.zip" | \
	docker run -v `pwd`/output:/output -i ${GIT_COMMIT} sh

# It's really there, right?  And it's really a zipfile, right?

[ -f output/function.zip ] && zip -T output/function.zip
