#!/bin/bash
set -uo pipefail

# Provides Aptly related functions. Aptly's API is hand rolled, and not the same as
# the API documented over at https://aptly.info. As such the opportunity to override
# APTLY_HOST should be considered carefully.

#export APTLY_HOST="aptly.internal.justin.tv:5000"
export APTLY_HOST="packages.internal.justin.tv"

# build - produces a debian artifact. Args are positional.
#         Arg 1: the project you're packaging up
#         Arg 2: the organization your project is part of
#         Arg 3: the version of your project
#         Arg 4: a canonical sha for your published artifact
#         Arg 5: the root of your build dir
#         Arg 6: the destination dir to output the artifact into
#
# Assumes that the build dir is a debian package structured dir.
#
# This function exports APTLY_ARTIFACT which points to the produced deb.
#
# This function will exit 1 if something goes wrong during packaging.
aptly::build() {
  local proj=${1}
  local org=${2}
  local version=${3}
  local sha=${4}
  local build_dir=${5}
  local output_dir=${6}

  # Replace all / and _ in the proj var with -
  proj=${proj//[\/_]/-}

  local artifact=${output_dir}/${org}-${proj}_${version}-${sha}.deb

  utils::check "packaging up ${org}/${proj} from ${build_dir} into ${artifact}" \
    dpkg-deb -Z gzip -b ${build_dir} ${artifact}

  export APTLY_ARTIFACT=${artifact}
}

# has_version - this function returns 0 if this version of the project has been published
#               to aptly, or 1 if it has not. Args are positional.
#               Arg 1: the project you're publishing an artifact to
#               Arg 2: the organization your project is part of
#               Arg 3: the version of your project
#               Arg 4: a canonical sha for your published artifact
aptly::has_version() {
  local proj=${1}
  local org=${2}
  local version=${3}
  local sha=${4}

  ###
  ### -- this is supposed to be the first letter of the package that we are uploading.  in this case it is
  ###    always going to be gds because our org is gds and the org name is always the first part of the package name.
  ###
  local firstLetter=${org:0:1}
  local dist="trusty"

  # Replace all / and _ in the proj var with -
  proj=${proj//[\/_]/-}

  ###
  ### -- find out if the file exists.  if it does not, $? will be 22 due to -f.
  ###
  curl --head -sf https://${APTLY_HOST}/artifactory/list/vr-internal/pool/main/${firstLetter}/${org}-${proj}/${org}-${proj}_${version}-${sha}_amd64.deb

  if [ $? -eq 0 ]; then
    return 0
  else
    return 1
  fi
}

# should_publish - returns 0 if this verion of the project has yet to be published. Args
#                  are positional
#                  Arg 1: the project you're publishing an artifact to
#                  Arg 2: the organization your project is part of
#                  Arg 3: the version of your project
#                  Arg 4: a canonical sha for your published artifact
aptly::should_publish() {
  local proj=${1}
  local org=${2}
  local version=${3}
  local sha=${4}

  aptly::has_version ${proj} ${org} ${version} ${sha}
  if [ $? -eq 1 ]; then
    return 0;
  fi

  echo "${org}/${proj} @ ${version}-${sha} already exists in aptly"
  return 1
}

# publish - publishes the provided artifact to APTLY_HOST. Arguments are positional.
#           Arg 1: The path to the artifact to publish
aptly::publish() {
  local artifact=${1}

  local package=$( dpkg-deb -I ${artifact} | grep "^ Package: " | cut -d" " -f3 )
  local version=$( dpkg-deb -I ${artifact} | grep "^ Version: " | cut -d" " -f3 )
  local arch=$( dpkg-deb -I ${artifact} | grep "^ Architecture: " | cut -d" " -f3 )
  ###
  ### -- this is supposed to be the first letter of the package that we are uploading.  in this case it is
  ###    always going to be gds because our org is gds and the org name is always the first part of the package name.
  ###
  local firstLetter=${package:0:1}
  local dist="trusty"

  ### dta_tools_deploy password is passed in to jenkins jobs with a
  ### credentialsBinding.
  ###
  ### Alternatively, you can set the environment variable
  ### ARTIFACTORY_API_KEY to the contents of your API key that you
  ### create as a logged-in user on packages.internal.justin.tv
  ###
  dta_tools_deploy=${dta_tools_deploy:-""}
  ARTIFACTORY_API_KEY=${ARTIFACTORY_API_KEY:-""}
  
  artifactory_auth="-u dta_tools:${dta_tools_deploy}"
  if [ "${ARTIFACTORY_API_KEY}" ]; then
	  artifactory_auth="-H X-JFrog-Art-Api:${ARTIFACTORY_API_KEY}"
  else
	  if [ -z "${dta_tools_deploy}" ]; then
		  echo "No credentials to upload to artifactory: neither of environment variables ARTIFACTORY_API_KEY or dta_tools_deploy are set"
		  return 1
	  fi
  fi
  utils::check "Publishing ${package}_${version}..." \
    curl ${artifactory_auth} -XPUT "http://${APTLY_HOST}/artifactory/ip-${dist}-twitch/pool/main/${firstLetter}/${package}/${package}_${version}_${arch}.deb;deb.architecture=${arch};deb.distribution=${dist};deb.component=main" \
	-T ${artifact}

  echo ""
  echo "Published ${artifact} @ ${version}"
}

# maybe_publish - calls aptly::publish if either of the following statements is true:
#                   * we have not published this version of the project to aptly yet
#
#                 Args are positional.
#                 Arg 1: the project you're publishing an artifact to
#                 Arg 2: the organization your project is part of
#                 Arg 3: the version of your project
#                 Arg 4: a canonical sha for your published artifact
#                 Arg 5: the artifact to publish
#
# This function exits 0 if the artifact is published, 1 otherwise.
aptly::maybe_publish() {
  local proj=${1}
  local org=${2}
  local version=${3}
  local sha=${4}
  local artifact=${5}

  aptly::should_publish ${proj} ${org} ${version} ${sha}
  if [ $? -eq 0 ]; then
    aptly::publish ${artifact}
    return 0
  fi

  return 1
}
