#!/bin/bash
set -uo pipefail

# Provides docker related functions. If you wish to target a different
# docker registry, after you source this file override DOCKER_FQDN. If
# you wish to use a different Dockerfile for your build you can override
# DOCKER_SERVICE after sourcing the file.

export DOCKER_FQDN="docker.internal.justin.tv"
export DOCKER_SERVICE=${DOCKER_SERVICE:-"Dockerfile-service"}

# build - builds the specified project. Args are positional.
#         Arg 1: the project which is being built. It is assumed this project has
#         a Dockerfile-service in it.
#         Arg 2: the organization your project is part of, this is used when publishing
#         your container, i.e. gds/project.
#         Arg 3: the tag you want to use, defaults to ${org}-${proj}-<rand>
#
# Invocation of this function assumes that your working directory contains the project
# directory, i.e. `pwd`/proj/Dockerfile-service.
# Invocation of this function takes place in the root of the git repo, with a context
# of '.'
#
# This function will export DOCKER_TAG and DOCKER_VERSION. It will exit 1 if your
# Dockerfile does not specify a twitch.version label.
docker::build() {
  local proj=${1}
  local org=${2}
  # Projdir is the directory where docker will build from as we're transforming $proj below
  local projdir=${proj}

  # we need the app for go builds to ensure we include all dependent code

  local app=$( echo ${projdir} | cut -d '/' -f1 )
  # Replace all / and _ in the proj var with -
  proj=${proj//[\/_]/-}

  local fqdn="${DOCKER_FQDN}/${org}/${proj}"
  local tag="${3:-${org}-${proj}-$$}"

  # we'll always use the `pwd` as the docker context
  local docker_context=$( pwd )

  # prep the docker ignore file
  utils::prepdockerignore ${projdir}

  for i in `grep ^FROM ${projdir}/${DOCKER_SERVICE} | awk '{print $2}'`; do 
    utils::check "docker pre-build pull of $i" \
      docker pull $i
  done

  utils::check "docker build" \
    docker build \
      --build-arg PROJDIR=${projdir} \
      --build-arg APP=${app} \
      -f ${projdir}/${DOCKER_SERVICE} -t ${fqdn}:${tag} ${docker_context}

  # PLUCK the twitch.version LABEL out of the image. See the file "Dockerfile-service" LABEL tag in the PROJ folder of interest
  local version=$(docker inspect -f "{{index .ContainerConfig.Labels \"twitch.version\"}}" ${fqdn}:${tag})

  if [ -z ${version} ]; then
    echo "Your docker image must specify a LABEL twitch.version=<some version>"
    exit 1
  fi

  echo "Version of build: ${version}"

  export DOCKER_VERSION=${version}
  export DOCKER_TAG=${tag}
  export DOCKER_CONTAINER=${fqdn}
}

# build_simple - builds the specified dockerfile
#         Arg 1: the organization your project is a part of, this is used when publishing
#         your container, i.e. ORG/project
#         Arg 2: the name of your image, this is used when publishing your container,
#         i.e gds/NAME
#         Arg 3: the dockerfile to reference during the build
#         Arg 4: the tag to use for this image
#	  Arg 5: the name of the project that the component being built is in
#
# Invocation of this function assumes your dockerfile is below the $PWD. If it is not this will
# fail because the dockerfile will not be within the docker context.
#
# This job is really intended as just a simple way to call docker build.
docker::build_simple() {
  local org=${1}
  local name=${2}
  local dockerfile=${3}
  local tag=${4}
  local proj=${5}

  local context=$(pwd)

  local app=$( echo ${projdir} | cut -d '/' -f1 )

  # prep the docker ignore file
  utils::prepdockerignore ${app}

  utils::check "building ${dockerfile} docker image" \
    docker build -t docker.internal.justin.tv/${org}/${name}:${tag} -f ${dockerfile} ${context}
}

# publish - publishes the local tagged image to the docker registry. Args are positional
#           Arg 1: the project which is being built.
#           Arg 2: the organization your project is part of
#           Arg 3: the tag of the image to publish
#
# It's assumed that the image has been tagged with the target docker registry.
#
# This function exports DOCKER_SHA as reported by the upstream docker registry
docker::publish() {
  local proj=${1}
  local org=${2}
  local tag=${3}

	# Replace all / and _ in the proj var with -
  proj=${proj//[\/_]/-}

  local fqdn="${DOCKER_FQDN}/${org}/${proj}"

  utils::check "publishing ${fqdn}:${tag}" \
    docker push ${fqdn}:${tag}

  local sha=$(docker pull ${fqdn}:${tag} | grep '^Digest' | cut -d':' -f 3)

  utils::check "fetching sha for ${fqdn}:${tag}" \
    test ! -z ${sha}

  echo "SHA for image: ${sha}"

  export DOCKER_SHA=${sha}
}
