#!/bin/bash
set -uo pipefail

# provides service related functions and properties. If you wish to use a different
# service template override SERVICE_TEMPLATE after sourcing this file. You can override
# SERVICE_OWNER to ensure published debs contain the correct team / email in them.
# SERVICE_POSTINST can be override to select a different file to execute as your postinstall
# script attached to the debian archive
# SERVICE_UPSTART can be overriden to point to a different upstart script.

export SERVICE_POSTINST=svc/docker_pull.sh
export SERVICE_UPSTART=svc/docker-compose.conf
export SERVICE_OWNER="Developer Success <ds@twitch.tv>"

# definition - produces a docker compose service file in the provided build_dir.
#              Args are positional.
#              Arg 1: the project which is being built
#              Arg 2: the organization your project is part of
#              Arg 3: the canonical sha for the published image
#              Arg 4: the root build dir to output the service.yaml into
#              Arg 5: the service.yaml template to use
#
# It is assumed that your build_dir has the following structure at a minimum:
#   - /
#     - opt/
#       - docker/
#
service::definition() {
  local proj=${1}
  local org=${2}
  local sha=${3}
  local build_dir=${4}
  local service_template=${5}

  # Replace all / and _ in the proj var with -
  proj=${proj//[\/_]/-}

  if [ ! -r ${service_template} ]; then
    echo "Couldn't find the service template ${service_template}"
    exit 1
  fi

  utils::check "setup service.yaml" \
    sed -e "s,_ORG_,${org},"        \
        -e "s,_TARGET_,${proj},"    \
        -e "s,_SHA_,sha256:${sha}," \
        ${service_template} > ${build_dir}/opt/docker/service.yaml
}

# controlfile - produces a debian control file in the build_dir provided. Args are positional
#               Arg 1: the project you're packaging up
#               Arg 2: the organization your project is part of
#               Arg 3: the version of your project
#               Arg 4: a canonical sha for your published artifact
#               Arg 5: the root of your build dir
#		Arg 6: the git commit that the docker image was built from
#
# It is assumed that your build_dir has the following structure at a minimum:
#   - /
#     - DEBIAN/
service::controlfile() {
  local proj=${1}
  local org=${2}
  local version=${3}
  local sha=${4}
  local build_dir=${5}
  local git_commit=${6}

  # Replace all / and _ in the proj var with -
  proj=${proj//[\/_]/-}

  cat > ${build_dir}/DEBIAN/control <<EOF
Package: ${org}-${proj}
Architecture: amd64
Section: admin
Maintainer: ${SERVICE_OWNER}
Homepage: https://code.justin.tv/${org}/${proj}
Description: ${org} ${proj} @ ${sha} for git commit ${git_commit}
Priority: standard
Version: ${version}-${sha}
EOF
}

# upstart - copies $SERVICE_UPSTART to build dir/etc/init/<nameoffile>.
#           The only arg is the build_dir
#
# It is assumed that your build_dir has the folowing structure at a minimum:
#   - /
#     - etc/
#       - init/
service::upstart() {
  local build_dir=${1}

  cp ${SERVICE_UPSTART} ${build_dir}/etc/init/
}

# postinst - copies $SERVICE_POSTINST to build dir/DEBIAN/postinst. The only arg is the build_dir
#
# It is assumed that your build_dir has the folowing structure at a minimum:
#   - /
#     - DEBIAN/
service::postinst() {
  local build_dir=${1}

  cp ${SERVICE_POSTINST} ${build_dir}/DEBIAN/postinst
}
