#!/bin/bash
set -uo pipefail

# Provides utilities used by other build tools

# jenkins build boxes have an old JQ, we provide our own
export JQ="${PWD}/bin/jq15-linux64"
if [ $(uname -s) == "Darwin" ]; then
  export JQ="jq"
fi

# check - helper function. Runs your command and in the case of a non-zero
#         exit code outputs the first argument and exits 1
utils::check() {
  local STAGE=${1}
  shift
  "$@"
  if [ $? -ne 0 ]; then
    echo "Failure during ${STAGE}"
    exit 1
  fi
}

# prepdirs - helper function which sets up our various directories. Expects an
#            argument pointing to the dir under which to put build and output files
#
# This function exports BUILD_DIR and OUTPUT_DIR as per prepbuilddir()
# and prepoutputdir() documentation
utils::prepdirs() {
  local pwd=${1}

  utils::prepbuilddir ${pwd}
  utils::prepoutputdir ${pwd}
}

# prepbuilddir - helperfunction which sets up our build dir
#
# this function exports BUILD_DIR which can be used to reference the
# build dir created by this function
utils::prepbuilddir() {
  local pwd=${1}

  local build_dir=${pwd}/build

  rm -rf ${build_dir}
  mkdir -p ${build_dir}/DEBIAN
  mkdir -p ${build_dir}/opt/docker
  mkdir -p ${build_dir}/etc/init

  export BUILD_DIR=${build_dir}
}

# prepbuilddir - helperfunction which sets up our build dir
#
# this function exports OUTPUT_DIR which can be used to reference the
# output dir required by our CI which expects output of successful runs to go to
# ${pwd}/output
utils::prepoutputdir() {
  local pwd=${1}

  local output_dir=${pwd}/output

  rm -rf ${output_dir}
  mkdir -p ${output_dir}

  export OUTPUT_DIR=${output_dir}
}

# prepdockerignore - helperfunction which sets up a root dockerignore file
#
# Our docker context is large, creating a dockerignore file shrinks it dramatically.
# projdir is the full directory path to the code being compiled, e.g. vhs/cmd/gatherer or
#	extensions/devassets-proxy.
utils::prepdockerignore() {
  local projdir=${1}
  bin/gen_dockerignore ${projdir}
}

utils::notnull() {
  local varname=${1}
  if [ $# -lt 2 -o -z "${2}" ]; then
    echo "Variable ${varname} is not set"
    exit 1
  fi
}
