#!/usr/bin/env bash
set -e

# Deploys Config Service to Dev / Prod cluster doing the following steps:
# * Push a docker image to ECR. By default is the image tagged with the current git commit sha (as tagged by make build scripts).
# * Use the script https://github.com/silinternational/ecs-deploy to update the ECS Task Definition with the new docker image, and then updates the ECS Service to use the new version. ECS will add new tasks first, which requires extra capacity on Ec2 tasks (Fargate provides its own capacity). Then, when the new task is healthy, stops one of the old tasks.
# * Wait until all tasks are restarted and working with the new version.
#
# Requirements:
#  * Build docker image first (see build_docker.sh script)
#  * AWS CLI (http://docs.aws.amazon.com/cli/latest/userguide/tutorial-ec2-ubuntu.html#install-cli)
#  * jq: install with: brew install jq
#  * mwinit (and access to Amazon VPN): https://wiki.twitch.com/display/AD/Using+Isengard+from+the+command+line+or+programatically
#  * ada credentials: install with: toolbox install ada

show_usage_and_exit() {
  echo "Usage: $0 <env> <docker-image-id>"
  echo "  <env>: dev|prod"
  echo "  <docker-image-id>: git-tag is the current git log commit by default, used for docker image tag"
  echo "Example: $0 dev docker-image"
  exit 1
}

ENV_NAME=$1
case $ENV_NAME in
  "dev")
    export AWS_PROFILE="twitch-extensions-configuration-dev"
    export AWS_ACCOUNT="925711164288"
    export ECR_REPO="devco-repos-1dh9idzjzlq3r"
    export CLUSTER="DevConfigService-ClusterEB0386A7-1UN53QWNROSO"
    export SERVICE="DevConfigService-ServiceD69D759B-12TOSQM2IRGJB"
    ;;
  "prod")
    export AWS_PROFILE="twitch-extensions-configuration-prod"
    export AWS_ACCOUNT="816111448886"
    export ECR_REPO="prodc-repos-1gt7rre4ypvub"
    export CLUSTER="ProdConfigService-ClusterEB0386A7-PHPDYKGYPGG6"
    export SERVICE="ProdConfigService-ServiceD69D759B-86GXU8HC8365"
    ;;
  *)
    echo "invalid <env>: $1"
    show_usage_and_exit
    ;;
esac

INPUT_IMAGE=$2

echo "Setting defaults"
PROFILE="--profile $AWS_PROFILE"
REGION="--region us-west-2"

echo "Getting latest git tag"
GIT_TAG=`git log -1 --pretty=%H`

echo "Getting ECR Repository"
REPOSITORY=$AWS_ACCOUNT.dkr.ecr.us-west-2.amazonaws.com/$ECR_REPO

echo "Login with ada. Make sure you are in Amazon VPN and ran mwinit before."
ada credentials update --once --provider=isengard --role=admin --profile=$AWS_PROFILE --account=$AWS_ACCOUNT

echo "Using Docker image: $INPUT_IMAGE"

pushd "$(dirname $0)"

DEPLOY_IMAGE_LATEST="${REPOSITORY}:latest"
DEPLOY_IMAGE_TAGGED="${REPOSITORY}:${GIT_TAG}"

echo "ECR get-login ..."
`aws ecr get-login --no-include-email ${PROFILE} ${REGION}`

echo "docker push $DEPLOY_IMAGE_TAGGED ..."
docker tag "$INPUT_IMAGE" "$DEPLOY_IMAGE_TAGGED"
docker push "$DEPLOY_IMAGE_TAGGED"

# Logic for ECS to update to newly pushed task definitions 
echo "Update Services to use new version ..."
bash ecs-deploy.sh ${PROFILE} ${REGION} --cluster "$CLUSTER" --service-name "$SERVICE"  --image "$DEPLOY_IMAGE_TAGGED" --timeout 60000

popd
