package api

import (
	"context"
	"testing"

	"code.justin.tv/extensions/configuration/services/main/auth"
	"code.justin.tv/extensions/configuration/services/main/data"
	"code.justin.tv/extensions/configuration/services/main/data/model"
	"code.justin.tv/extensions/configuration/services/main/data/model/bad"
	"code.justin.tv/extensions/configuration/services/main/data/model/memory"
	"code.justin.tv/extensions/configuration/services/main/protocol"
	"code.justin.tv/gds/gds/golibs/params"
	"code.justin.tv/gds/gds/golibs/uuid"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestDeleteConfiguration(t *testing.T) {
	a := NewAPI(0, nil)

	t.Run("should report unimplemented without a manager", func(t *testing.T) {
		out, err := a.DeleteConfiguration(context.Background())
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrUnimplemented, err)
	})

	t.Run("should report if channel is missing", func(t *testing.T) {
		ctx := data.Store(context.Background(), createManager(nil))
		out, err := a.DeleteConfiguration(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrMissingChannelID, err)
	})

	t.Run("should report unauthorized when applicable", func(t *testing.T) {
		ctx := params.Set(context.Background(), "cid", "ch")
		ctx = data.Store(ctx, createManager(nil))
		out, err := a.DeleteConfiguration(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrUnauthorized, err)
	})

	t.Run("should succeed if already empty", func(t *testing.T) {
		ctx := params.Set(context.Background(), "cid", "ch")
		ctx = auth.Store(ctx, auth.AllPermissions())
		ctx = data.Store(ctx, createManager(nil))
		out, err := a.DeleteConfiguration(ctx)
		assert.Nil(t, out)
		assert.NoError(t, err)
	})

	t.Run("should succeed if not empty", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		ch := model.NewChannel(model.DefaultEnvironment, "extID", "chID")
		ch.Developer = protocol.NewRecord("2", nil)
		require.NoError(t, store.SaveChannel(ch))

		ctx := params.Set(context.Background(), "cid", "chID")
		ctx = auth.Store(ctx, &auth.FakeCredentials{BroadcasterChannel: "chID"})
		ctx = data.Store(ctx, createManager(store))
		out, err := a.DeleteConfiguration(ctx)
		assert.Nil(t, out)
		assert.NoError(t, err)

		ch, err = store.LoadChannel(model.DefaultEnvironment, "extID", "chID")
		assert.Nil(t, ch)
		assert.NoError(t, err)
	})

	t.Run("should forward store errors", func(t *testing.T) {
		ctx := params.Set(context.Background(), "cid", "chID")
		ctx = auth.Store(ctx, &auth.FakeCredentials{BroadcasterChannel: "chID"})
		ctx = data.Store(ctx, createManager(bad.New()))
		out, err := a.DeleteConfiguration(ctx)
		assert.Nil(t, out)
		assert.Equal(t, bad.ErrExpected, err)
	})
}
