package api

import (
	"context"
	"testing"

	"code.justin.tv/extensions/configuration/services/main/data"
	"code.justin.tv/extensions/configuration/services/main/data/model"
	"code.justin.tv/extensions/configuration/services/main/data/model/bad"
	"code.justin.tv/extensions/configuration/services/main/data/model/memory"
	"code.justin.tv/extensions/configuration/services/main/protocol"
	"code.justin.tv/gds/gds/golibs/errors"
	"code.justin.tv/gds/gds/golibs/params"
	"code.justin.tv/gds/gds/golibs/uuid"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestGetChannelConfiguration(t *testing.T) {
	a := NewAPI(0, nil)

	t.Run("should report unimplemented without a manager", func(t *testing.T) {
		out, err := a.GetChannelConfiguration(context.Background())
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrUnimplemented, err)
	})

	t.Run("should report if channel is missing", func(t *testing.T) {
		ctx := data.Store(context.Background(), createManager(nil))
		out, err := a.GetChannelConfiguration(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrIllegalSegmentChannelCode, errors.GetErrorCode(err))
	})

	t.Run("should report unimplemented without manager", func(t *testing.T) {
		ctx := params.Set(context.Background(), "cid", "ch")
		out, err := a.GetChannelConfiguration(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrUnimplemented, err)
	})

	t.Run("should succeed if request is empty", func(t *testing.T) {
		ctx := params.Set(context.Background(), "cid", "ch")
		ctx = data.Store(ctx, createManager(nil))
		out, err := a.GetChannelConfiguration(ctx)
		assert.Empty(t, out)
		assert.NoError(t, err)
	})

	t.Run("should succeed if not found", func(t *testing.T) {
		ctx := params.Set(context.Background(), "cid", "chID")
		ctx = params.Set(ctx, "ext_ids", "extID")
		ctx = data.Store(ctx, createManager(nil))
		out, err := a.GetChannelConfiguration(ctx)
		assert.Empty(t, out)
		assert.NoError(t, err)
	})

	t.Run("should succeed if partially found", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		ch := model.NewChannel(model.DefaultEnvironment, "extID", "chID")
		ch.Broadcaster = protocol.NewRecord("2", nil)
		require.NoError(t, store.SaveChannel(ch))

		co := model.NewCommon(ch.Environment, ch.ExtensionID)
		co.Global = protocol.NewRecord("1", nil)
		require.NoError(t, store.SaveCommon(co))

		ctx := params.Set(context.Background(), "cid", ch.ChannelID)
		ctx = params.Set(ctx, "ext_ids", ch.ExtensionID)
		ctx = data.Store(ctx, createManager(store))
		out, err := a.GetChannelConfiguration(ctx)
		assert.Len(t, out, 2) // should find extID broadcaster and global
		assert.NoError(t, err)
	})

	t.Run("should succeed for multiple extensions", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		ch := model.NewChannel(model.DefaultEnvironment, "extID", "chID")
		ch.Broadcaster = protocol.NewRecord("2", nil)
		require.NoError(t, store.SaveChannel(ch))

		co := model.NewCommon(ch.Environment, "ext2ID")
		co.Global = protocol.NewRecord("1", nil)
		require.NoError(t, store.SaveCommon(co))

		ctx := params.Set(context.Background(), "cid", ch.ChannelID)
		ctx = params.Set(ctx, "ext_ids", "extID, ext2ID")
		ctx = data.Store(ctx, createManager(store))
		out, err := a.GetChannelConfiguration(ctx)
		assert.Len(t, out, 2) // should find extID broadcaster and global
		assert.NoError(t, err)
	})

	t.Run("should forward store errors", func(t *testing.T) {
		ctx := params.Set(context.Background(), "cid", "ch")
		ctx = data.Store(ctx, createManager(bad.New()))
		out, err := a.GetChannelConfiguration(ctx)
		assert.Nil(t, out)
		assert.Equal(t, bad.ErrExpected, err)
	})

	t.Run("should fail if common is not a bool", func(t *testing.T) {
		ctx := params.Set(context.Background(), "cid", "cid")
		ctx = params.Set(ctx, "ext_ids", "extID, ext2ID")
		ctx = params.Set(ctx, "common", "blah")
		out, err := a.GetChannelConfiguration(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrInvalidCommonParameter, err)
	})

	t.Run("should not return common when common is false", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		ch := model.NewChannel(model.DefaultEnvironment, "extID", "chID")
		ch.Broadcaster = protocol.NewRecord("2", nil)
		require.NoError(t, store.SaveChannel(ch))

		co := model.NewCommon(ch.Environment, "ext2ID")
		co.Global = protocol.NewRecord("1", nil)
		require.NoError(t, store.SaveCommon(co))

		ctx := params.Set(context.Background(), "cid", ch.ChannelID)
		ctx = params.Set(ctx, "ext_ids", "extID, ext2ID")
		ctx = params.Set(ctx, "common", "false")
		ctx = data.Store(ctx, createManager(store))
		out, err := a.GetChannelConfiguration(ctx)
		assert.Len(t, out, 1) // should find extID broadcaster and not global
		assert.NoError(t, err)
	})
}
