package api

import (
	"context"
	"testing"

	"code.justin.tv/extensions/configuration/services/main/auth"
	"code.justin.tv/extensions/configuration/services/main/data"
	"code.justin.tv/extensions/configuration/services/main/data/model"
	"code.justin.tv/extensions/configuration/services/main/data/model/bad"
	"code.justin.tv/extensions/configuration/services/main/data/model/memory"
	"code.justin.tv/extensions/configuration/services/main/protocol"
	"code.justin.tv/gds/gds/golibs/params"
	"code.justin.tv/gds/gds/golibs/uuid"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestGetConfiguration(t *testing.T) {
	a := NewAPI(0, nil)

	t.Run("should report missing body", func(t *testing.T) {
		out, err := a.GetConfiguration(context.Background())
		assert.Nil(t, out)
		assert.Equal(t, params.ErrMissingBody, err)
	})

	t.Run("should report unimplemented without manager", func(t *testing.T) {
		ctx := createBody(context.Background(), []protocol.Address{})
		out, err := a.GetConfiguration(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrUnimplemented, err)
	})

	t.Run("should succeed if request is empty", func(t *testing.T) {
		ctx := createBody(context.Background(), []protocol.Address{})
		ctx = data.Store(ctx, createManager(nil))
		out, err := a.GetConfiguration(ctx)
		assert.Empty(t, out)
		assert.NoError(t, err)
	})

	t.Run("should succeed if not found", func(t *testing.T) {
		ctx := createBody(context.Background(), []protocol.Address{
			protocol.Address{
				ExtensionID: "extID",
				ChannelID:   "chID",
				SegmentType: protocol.GlobalType,
			},
		})
		ctx = auth.Store(ctx, auth.AllPermissions())
		ctx = data.Store(ctx, createManager(nil))
		out, err := a.GetConfiguration(ctx)
		assert.Empty(t, out)
		assert.NoError(t, err)
	})

	t.Run("should succeed if partially found", func(t *testing.T) {
		store := memory.New(uuid.NewSource())
		ch := model.NewChannel(model.DefaultEnvironment, "extID", "chID")
		ch.Broadcaster = protocol.NewRecord("2", nil)
		require.NoError(t, store.SaveChannel(ch))

		ctx := createBody(context.Background(), []protocol.Address{
			protocol.Address{
				ExtensionID: "extID",
				ChannelID:   "chID",
				SegmentType: protocol.BroadcasterType,
			},
			protocol.Address{
				ExtensionID: "ext2ID",
				ChannelID:   "chID",
				SegmentType: protocol.BroadcasterType,
			},
		})
		ctx = data.Store(ctx, createManager(store))
		out, err := a.GetConfiguration(ctx)
		assert.Len(t, out, 1) // should find extID
		assert.NoError(t, err)
	})

	t.Run("should forward store errors", func(t *testing.T) {
		ctx := createBody(context.Background(), []protocol.Address{
			protocol.Address{
				ExtensionID: "extID",
				ChannelID:   "chID",
				SegmentType: protocol.BroadcasterType,
			},
			protocol.Address{
				ExtensionID: "ext2ID",
				ChannelID:   "chID",
				SegmentType: protocol.BroadcasterType,
			},
		})
		ctx = data.Store(ctx, createManager(bad.New()))
		out, err := a.GetConfiguration(ctx)
		assert.Nil(t, out)
		assert.Equal(t, bad.ErrExpected, err)
	})
}
