package api

import (
	"context"
	"testing"

	"code.justin.tv/extensions/configuration/services/main/auth"
	"code.justin.tv/extensions/configuration/services/main/data"
	"code.justin.tv/extensions/configuration/services/main/data/model"
	"code.justin.tv/extensions/configuration/services/main/data/model/bad"
	"code.justin.tv/extensions/configuration/services/main/data/model/memory"
	"code.justin.tv/extensions/configuration/services/main/protocol"
	"code.justin.tv/gds/gds/golibs/errors"
	"code.justin.tv/gds/gds/golibs/params"
	"code.justin.tv/gds/gds/golibs/uuid"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestSetConfigurationV2(t *testing.T) {
	a := NewAPI(0, nil)

	t.Run("should report missing body", func(t *testing.T) {
		out, err := a.SetConfigurationV2(context.Background())
		assert.Nil(t, out)
		assert.Equal(t, params.ErrMissingBody, err)
	})

	t.Run("should report unknown segment type", func(t *testing.T) {
		ctx := createBody(context.Background(), &protocol.SetConfigurationInputV2{
			ExtJwt:   fakeJwt,
			ClientId: fakeClientId,
			UserID:   fakeUserId,
		})
		out, err := a.SetConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrUnknownSegmentTypeCode, errors.GetErrorCode(err))
	})

	t.Run("should report illegal segment channel", func(t *testing.T) {
		ctx := createBody(context.Background(), &protocol.SetConfigurationInputV2{
			Address: protocol.Address{
				SegmentType: protocol.BroadcasterType,
			},
			ExtJwt:   fakeJwt,
			ClientId: fakeClientId,
			UserID:   fakeUserId,
		})
		out, err := a.SetConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrIllegalSegmentChannelCode, errors.GetErrorCode(err))
	})

	t.Run("should report unimplemented without manager", func(t *testing.T) {
		ctx := createBody(context.Background(), &protocol.SetConfigurationInputV2{
			Address: protocol.Address{
				SegmentType: protocol.BroadcasterType,
				ChannelID:   "chID",
			},
			ExtJwt:   fakeJwt,
			ClientId: fakeClientId,
			UserID:   fakeUserId,
		})
		out, err := a.SetConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrUnimplemented, err)
	})

	t.Run("should report missing extension ID", func(t *testing.T) {
		ctx := createBody(context.Background(), &protocol.SetConfigurationInputV2{
			Address: protocol.Address{
				SegmentType: protocol.BroadcasterType,
				ChannelID:   "chID",
			},
			ExtJwt:   fakeJwt,
			ClientId: fakeClientId,
			UserID:   fakeUserId,
		})
		ctx = data.Store(ctx, createManager(nil))
		out, err := a.SetConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrMissingExtensionIDCode, errors.GetErrorCode(err))
	})

	t.Run("should require authentication", func(t *testing.T) {
		ctx := createBody(context.Background(), &protocol.SetConfigurationInputV2{
			Address: protocol.Address{
				SegmentType: protocol.BroadcasterType,
				ExtensionID: "extID",
				ChannelID:   "chID",
			},
			ExtJwt:   fakeJwt,
			ClientId: fakeClientId,
			UserID:   fakeUserId,
		})
		ctx = data.Store(ctx, createManager(bad.New()))
		out, err := a.SetConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrUnauthorized, err)
	})

	t.Run("should report illegal length", func(t *testing.T) {
		ctx := createBody(context.Background(), &protocol.SetConfigurationInputV2{
			Address: protocol.Address{
				SegmentType: protocol.BroadcasterType,
				ExtensionID: "extID",
				ChannelID:   "chID",
			},
			ExtJwt:   fakeJwt,
			ClientId: fakeClientId,
			UserID:   fakeUserId,
			Record:   protocol.NewRecord("", makeLongString(protocol.MaxSegmentLength+1)),
		})
		ctx = auth.Store(ctx, auth.AllPermissions())
		ctx = data.Store(ctx, createManager(bad.New()))
		out, err := a.SetConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrIllegalSegmentLengthCode, errors.GetErrorCode(err))
	})

	t.Run("should succeed if created", func(t *testing.T) {
		addr := protocol.Address{
			SegmentType: protocol.BroadcasterType,
			ExtensionID: "extID",
			ChannelID:   "chID",
		}
		store := memory.New(uuid.NewSource())
		ctx := createBody(context.Background(), &protocol.SetConfigurationInputV2{
			Address:  addr,
			Record:   protocol.NewRecord("2", nil),
			ExtJwt:   fakeJwt,
			ClientId: fakeClientId,
			UserID:   fakeUserId,
		})
		ctx = auth.Store(ctx, auth.AllPermissions())
		ctx = data.Store(ctx, createManager(store))
		out, err := a.SetConfigurationV2(ctx)
		assert.Nil(t, out)
		assert.NoError(t, err)
		ch, err := store.LoadChannel(model.DefaultEnvironment, addr.ExtensionID, addr.ChannelID)
		assert.NoError(t, err)
		require.NotNil(t, ch)
		assert.NotNil(t, ch.Broadcaster)
		assert.Equal(t, "2", ch.Broadcaster.Version())
	})

	t.Run("should succeed if updating", func(t *testing.T) {
		addr := protocol.Address{
			SegmentType: protocol.BroadcasterType,
			ExtensionID: "extID",
			ChannelID:   "chID",
		}
		store := memory.New(uuid.NewSource())
		ch := model.NewChannel(model.DefaultEnvironment, addr.ExtensionID, addr.ChannelID)
		ch.Broadcaster = protocol.NewRecord("1", nil)
		require.NoError(t, store.SaveChannel(ch))

		ctx := createBody(context.Background(), &protocol.SetConfigurationInputV2{
			Address:  addr,
			Record:   protocol.NewRecord("3", nil),
			ExtJwt:   fakeJwt,
			ClientId: fakeClientId,
			UserID:   fakeUserId,
		})
		ctx = auth.Store(ctx, auth.AllPermissions())
		ctx = data.Store(ctx, createManager(store))
		out, err := a.SetConfiguration(ctx)
		assert.Nil(t, out)
		assert.NoError(t, err)
		ch, err = store.LoadChannel(model.DefaultEnvironment, addr.ExtensionID, addr.ChannelID)
		assert.NoError(t, err)
		require.NotNil(t, ch)
		require.NotNil(t, ch.Broadcaster)
		assert.Equal(t, "3", ch.Broadcaster.Version())
	})

	t.Run("should forward store errors", func(t *testing.T) {
		ctx := createBody(context.Background(), &protocol.SetConfigurationInputV2{
			Address: protocol.Address{
				SegmentType: protocol.BroadcasterType,
				ExtensionID: "extID",
				ChannelID:   "chID",
			},
			ExtJwt:   fakeJwt,
			ClientId: fakeClientId,
			UserID:   fakeUserId,
		})
		ctx = auth.Store(ctx, auth.AllPermissions())
		ctx = data.Store(ctx, createManager(bad.New()))
		out, err := a.SetConfiguration(ctx)
		assert.Nil(t, out)
		assert.Equal(t, bad.ErrExpected, err)
	})
}
