package auth

import (
	"code.justin.tv/extensions/configuration/services/main/protocol"
	"context"
	"fmt"
	"github.com/afex/hystrix-go/hystrix"
	statsdClient "github.com/cactus/go-statsd-client/statsd"
	"github.com/sirupsen/logrus"
	"reflect"
	"strconv"
	"time"

	barbradytwirp "code.justin.tv/amzn/TwitchExtensionsBarbradyTwirp"
)

type BarbradyCredentials struct {
	alreadyCompared     map[string]bool
	capabilities        []string
	client              barbradytwirp.TwitchExtensionsBarbrady
	endpoint            string
	fallbackCredentials Credentials
	grantedCapabilities map[string]*barbradytwirp.Capability
	parameters          map[string]string
	shouldCompare       bool
	statsd              statsdClient.Statter
}

func NewBarbradyCredentials(fallbackCredentials Credentials, barbradyClient barbradytwirp.TwitchExtensionsBarbrady, statsd statsdClient.Statter) Credentials {
	return &BarbradyCredentials{
		alreadyCompared:     map[string]bool{},
		fallbackCredentials: fallbackCredentials,
		client:              barbradyClient,
		grantedCapabilities: map[string]*barbradytwirp.Capability{},
		shouldCompare:       true,
		statsd:              statsd,
	}
}

func (b BarbradyCredentials) UserID() *string {
	return b.fallbackCredentials.UserID()
}

func (b BarbradyCredentials) ClientID() string {
	return b.fallbackCredentials.ClientID()
}

func (b BarbradyCredentials) CanDeleteConfig(chID string) bool {
	barbradyGranted := b.BarbradyCanDeleteConfig(chID)
	fallbackGranted := b.fallbackCredentials.CanDeleteConfig(chID)

	b.compare("CanDeleteConfig", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b BarbradyCredentials) BarbradyCanDeleteConfig(chID string) bool {
	return b.checkGrantedCapabilities(CapEditBroadcasterConfig, &map[string]string{ParamChannelID: chID})
}

func (b BarbradyCredentials) CanReadConfig(extID string, segment protocol.Segment) bool {
	barbradyGranted := b.BarbradyCanReadConfig(extID, segment)
	fallbackGranted := b.fallbackCredentials.CanReadConfig(extID, segment)

	b.compare("CanReadConfig", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b BarbradyCredentials) BarbradyCanReadConfig(extID string, segment protocol.Segment) bool {
	return b.checkGrantedCapabilities(CapReadConfig, &map[string]string{})
}

func (b BarbradyCredentials) CanEditConfig(extID string, segment protocol.Segment) bool {
	barbradyGranted := b.BarbradyCanEditConfig(extID, segment)
	fallbackGranted := b.fallbackCredentials.CanEditConfig(extID, segment)

	b.compare("CanEditConfig", barbradyGranted, fallbackGranted)

	return fallbackGranted
}

func (b BarbradyCredentials) BarbradyCanEditConfig(extID string, segment protocol.Segment) bool {
	if segment.Type() == protocol.BroadcasterType {
		return b.checkGrantedCapabilities(CapEditBroadcasterConfig, &map[string]string{ParamChannelID: segment.ChannelID()})
	}
	return b.checkGrantedCapabilities(CapEditDeveloperConfig, &map[string]string{ParamExtensionID: extID})
}

// Barbrady Credentials Specific Code

func (b *BarbradyCredentials) RequestCapabilities(capabilities []string, params map[string]string) {
	b.capabilities = capabilities

	b.parameters = params
	// Merge UserID or ClientID into params
	if b.parameters[ParamClientID] == "" && b.parameters[ParamUserID] == "" {
		if b.UserID() != nil {
			b.parameters[ParamUserID] = *b.UserID()
		} else {
			b.parameters[ParamClientID] = b.ClientID()
		}
	}

	var resp *barbradytwirp.GetCapabilitiesResponse
	var respErr error

	_ = b.statsd.Inc("Barbrady_GetCapabilities", 1, 1.0)
	err := hystrix.Do("barbrady_capability_lookup", func() error {
		start := time.Now()
		resp, respErr = b.client.GetCapabilities(context.Background(), &barbradytwirp.GetCapabilitiesRequest{
			Capabilities: b.capabilities,
			Parameters:   b.parameters,
		})
		duration := time.Since(start)
		_ = b.statsd.TimingDuration("Barbrady_GetCapabilities_Duration", duration, 1.0)
		return respErr
	}, nil)

	if err != nil {
		_ = b.statsd.Inc("Barbrady_GetCapabilities_Error", 1, 1.0)
		logrus.WithFields(logrus.Fields{
			"capabilities": b.capabilities,
			"parameters":   b.parameters,
			"endpoint":     b.endpoint,
			"error":        err,
		}).Error("Error Getting Capabilities from Barbrady")

		b.shouldCompare = false

		return
	}

	for _, capability := range resp.Capabilities {
		b.grantedCapabilities[capability.Name] = capability
	}
}

func (b *BarbradyCredentials) String() string {
	return fmt.Sprintf("%+v", b.grantedCapabilities)
}

func (b *BarbradyCredentials) FallbackCredentials() Credentials {
	return b.fallbackCredentials
}

func (b *BarbradyCredentials) SetEndpoint(endpoint string) {
	b.endpoint = endpoint
}

func (b *BarbradyCredentials) compare(method string, barbradyIsGranted, cartmanIsGranted bool) {
	if !b.isSourceCartman() && !b.alreadyCompared[method] && b.shouldCompare {
		b.alreadyCompared[method] = true

		if barbradyIsGranted != cartmanIsGranted {
			b.logDifference(method, strconv.FormatBool(barbradyIsGranted), strconv.FormatBool(cartmanIsGranted))
		} else {
			_ = b.statsd.Inc(fmt.Sprintf("Barbrady_Compare_Match_%s", method), 1, 1.0)
		}
	}
}

func (b *BarbradyCredentials) logDifference(method, barbradyIsGranted, cartmanIsGranted string) {
	_ = b.statsd.Inc(fmt.Sprintf("Barbrady_Compare_Mismatch_%s", method), 1, 1.0)
	logrus.WithFields(logrus.Fields{
		"authorization_method":   method,
		"barbrady_capabilities":  b.String(),
		"barbrady_is_granted":    barbradyIsGranted,
		"endpoint":               b.endpoint,
		"fallback_is_granted":    cartmanIsGranted,
		"fallback_capabilities":  b.fallbackCredentials.String(),
		"fallback_type":          reflect.TypeOf(b.fallbackCredentials),
		"parameters":             b.parameters,
		"requested_capabilities": b.capabilities,
	}).Warn("Barbrady Authorization did not match Fallback Credentials!")
}

func (b *BarbradyCredentials) checkGrantedCapabilities(capability string, requiredParams *map[string]string) bool {
	if cap := b.grantedCapabilities[capability]; cap != nil && cap.Granted {
		if requiredParams != nil {
			for key, value := range *requiredParams {
				if b.parameters[key] != value {
					return false
				}
			}
		}

		return true
	}
	return false
}

func (b *BarbradyCredentials) isSourceCartman() bool {
	return b.UserID() == nil && b.ClientID() == ""
}
