package auth

import (
	barbradytwirp "code.justin.tv/amzn/TwitchExtensionsBarbradyTwirp"
	"code.justin.tv/extensions/configuration/services/main/auth/fakes"
	"code.justin.tv/extensions/configuration/services/main/protocol"
	statsdClient "github.com/cactus/go-statsd-client/statsd"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
	"testing"
)

const (
	fakeUserIdValue = "1"
	fakeClientId    = ""
	fakeChannelId   = "5"
	fakeExtensionId = "10"

	fakeIncorrectChannelId = "100"
)

func Test_BarbradyCredentials(t *testing.T) {
	fakeCartmanCredentials := &FakeCredentials{UserIDValue: fakeUserIdValue}
	fakeBarbradyClient := &fakes.FakeTwitchExtensionsBarbrady{}

	fakeDeveloperSegment, err := protocol.Developer(fakeChannelId)
	require.Nil(t, err)
	fakeBroadcasterSegment, err := protocol.Broadcaster(fakeChannelId)
	require.Nil(t, err)

	mockStatsd, err := statsdClient.NewClient(":8000", "mock_prefix")
	require.Nil(t, err)
	require.NotNil(t, mockStatsd)

	runGeneralBarbradyCredentialTests(t, fakeCartmanCredentials, fakeBarbradyClient, mockStatsd)
	runBarbradyCanDeleteConfigTests(t, fakeCartmanCredentials, fakeBarbradyClient, mockStatsd)
	runBarbradyCanEditConfigTests(t, fakeCartmanCredentials, fakeBarbradyClient, mockStatsd, fakeDeveloperSegment, fakeBroadcasterSegment)
	runBarbradyCanReadConfigTests(t, fakeCartmanCredentials, fakeBarbradyClient, mockStatsd, fakeDeveloperSegment)
}

func runGeneralBarbradyCredentialTests(t *testing.T, fakeCartmanCredentials *FakeCredentials, fakeBarbradyClient *fakes.FakeTwitchExtensionsBarbrady, mockStatsd statsdClient.Statter) {
	t.Run("userID returns cartman creds userID", func(t *testing.T) {
		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		userId := barbradyCredentials.UserID()
		assert.Equal(t, fakeUserIdValue, *userId)
	})

	t.Run("clientID returns cartman creds clientID", func(t *testing.T) {
		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		clientId := barbradyCredentials.ClientID()
		assert.Equal(t, fakeClientId, clientId)
	})
}

func runBarbradyCanDeleteConfigTests(t *testing.T, fakeCartmanCredentials *FakeCredentials, fakeBarbradyClient *fakes.FakeTwitchExtensionsBarbrady, mockStatsd statsdClient.Statter) {
	t.Run("return true if user is granted capability with matching params", func(t *testing.T) {
		params := map[string]string{ParamChannelID: fakeChannelId}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{
				{
					Name:       CapEditBroadcasterConfig,
					Granted:    true,
					Parameters: params,
				},
			},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapEditBroadcasterConfig}, params)
		granted := barbradyCredentials.BarbradyCanDeleteConfig(fakeChannelId)
		assert.True(t, granted)
	})

	t.Run("return false if user is granted capability with different params", func(t *testing.T) {
		params := map[string]string{ParamChannelID: fakeChannelId}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{
				{
					Name:       CapEditBroadcasterConfig,
					Granted:    true,
					Parameters: params,
				},
			},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapEditBroadcasterConfig}, params)
		granted := barbradyCredentials.BarbradyCanDeleteConfig(fakeIncorrectChannelId)
		assert.False(t, granted)
	})

	t.Run("return false if user was not granted capability", func(t *testing.T) {
		params := map[string]string{ParamChannelID: fakeChannelId}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{
				{
					Name:       CapEditBroadcasterConfig,
					Granted:    false,
					Parameters: params,
				},
			},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapEditBroadcasterConfig}, params)
		granted := barbradyCredentials.BarbradyCanDeleteConfig(fakeChannelId)
		assert.False(t, granted)
	})

	t.Run("return false if capability not returned from barbrady", func(t *testing.T) {
		params := map[string]string{ParamChannelID: fakeChannelId}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapEditBroadcasterConfig}, params)
		granted := barbradyCredentials.BarbradyCanDeleteConfig(fakeChannelId)
		assert.False(t, granted)
	})

	t.Run("return fallback creds if Barbrady returns something different", func(t *testing.T) {
		allPermissionsCartmanCreds := AllPermissions()
		params := map[string]string{ParamChannelID: fakeChannelId}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(allPermissionsCartmanCreds, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapEditBroadcasterConfig}, params)

		barbradyCredsGranted := barbradyCredentials.BarbradyCanDeleteConfig(fakeChannelId)
		fallbackGranted := allPermissionsCartmanCreds.CanDeleteConfig(fakeChannelId)
		credsGranted := barbradyCredentials.CanDeleteConfig(fakeChannelId)
		assert.False(t, barbradyCredsGranted)
		assert.True(t, fallbackGranted)
		assert.NotEqual(t, credsGranted, barbradyCredsGranted)
		assert.Equal(t, credsGranted, fallbackGranted)
	})
}

func runBarbradyCanEditConfigTests(t *testing.T, fakeCartmanCredentials *FakeCredentials, fakeBarbradyClient *fakes.FakeTwitchExtensionsBarbrady, mockStatsd statsdClient.Statter, developerSegment protocol.Segment, broadcasterSegment protocol.Segment) {
	t.Run("return true for broadcaster type if user is granted params", func(t *testing.T) {
		params := map[string]string{ParamChannelID: fakeChannelId}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{
				{
					Name:       CapEditBroadcasterConfig,
					Granted:    true,
					Parameters: params,
				},
			},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapEditBroadcasterConfig}, params)
		granted := barbradyCredentials.BarbradyCanEditConfig(fakeExtensionId, broadcasterSegment)
		assert.True(t, granted)
	})

	t.Run("return true for developer type if user is granted params", func(t *testing.T) {
		params := map[string]string{ParamExtensionID: fakeExtensionId}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{
				{
					Name:       CapEditDeveloperConfig,
					Granted:    true,
					Parameters: params,
				},
			},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapEditDeveloperConfig}, params)
		granted := barbradyCredentials.BarbradyCanEditConfig(fakeExtensionId, developerSegment)
		assert.True(t, granted)
	})

	t.Run("return false if user is granted capability with different params for broadcaster type", func(t *testing.T) {
		params := map[string]string{ParamChannelID: fakeChannelId}
		fakeIncorrectBroadcasterSegment, err := protocol.Broadcaster(fakeIncorrectChannelId)
		require.Nil(t, err)

		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{
				{
					Name:       CapEditBroadcasterConfig,
					Granted:    true,
					Parameters: params,
				},
			},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapEditBroadcasterConfig}, params)
		granted := barbradyCredentials.BarbradyCanEditConfig(extID, fakeIncorrectBroadcasterSegment)
		assert.False(t, granted)
	})

	t.Run("return false if user is granted capability with different params for developer type", func(t *testing.T) {
		params := map[string]string{ParamExtensionID: fakeExtensionId}
		fakeIncorrectDeveloperSegment, err := protocol.Developer(fakeIncorrectChannelId)
		require.Nil(t, err)

		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{
				{
					Name:       CapEditDeveloperConfig,
					Granted:    true,
					Parameters: params,
				},
			},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapEditDeveloperConfig}, params)
		granted := barbradyCredentials.BarbradyCanEditConfig(extID, fakeIncorrectDeveloperSegment)
		assert.False(t, granted)
	})

	t.Run("return false if user was not granted capability for broadcaster", func(t *testing.T) {
		params := map[string]string{ParamChannelID: fakeChannelId}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{
				{
					Name:       CapEditBroadcasterConfig,
					Granted:    false,
					Parameters: params,
				},
			},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapEditBroadcasterConfig}, params)
		granted := barbradyCredentials.BarbradyCanEditConfig(fakeExtensionId, broadcasterSegment)
		assert.False(t, granted)
	})

	t.Run("return false if user was not granted capability for developer", func(t *testing.T) {
		params := map[string]string{ParamExtensionID: fakeExtensionId}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{
				{
					Name:       CapEditDeveloperConfig,
					Granted:    false,
					Parameters: params,
				},
			},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapEditDeveloperConfig}, params)
		granted := barbradyCredentials.BarbradyCanEditConfig(fakeExtensionId, developerSegment)
		assert.False(t, granted)
	})

	t.Run("return false if capability not returned from barbrady for broadcaster", func(t *testing.T) {
		params := map[string]string{ParamChannelID: fakeChannelId}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapEditBroadcasterConfig}, params)
		granted := barbradyCredentials.BarbradyCanEditConfig(fakeExtensionId, broadcasterSegment)
		assert.False(t, granted)
	})

	t.Run("return false if capability not returned from barbrady for developer", func(t *testing.T) {
		params := map[string]string{ParamExtensionID: fakeExtensionId}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapEditBroadcasterConfig}, params)
		granted := barbradyCredentials.BarbradyCanEditConfig(fakeExtensionId, developerSegment)
		assert.False(t, granted)
	})

	t.Run("return fallback creds if Barbrady returns something different for broadcaster", func(t *testing.T) {
		allPermissionsCartmanCreds := AllPermissions()
		params := map[string]string{ParamChannelID: fakeChannelId}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(allPermissionsCartmanCreds, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapEditBroadcasterConfig}, params)

		barbradyCredsGranted := barbradyCredentials.BarbradyCanEditConfig(fakeExtensionId, broadcasterSegment)
		fallbackGranted := allPermissionsCartmanCreds.CanEditConfig(fakeExtensionId, broadcasterSegment)
		credsGranted := barbradyCredentials.CanEditConfig(fakeExtensionId, broadcasterSegment)
		assert.False(t, barbradyCredsGranted)
		assert.True(t, fallbackGranted)
		assert.NotEqual(t, credsGranted, barbradyCredsGranted)
		assert.Equal(t, credsGranted, fallbackGranted)
	})

	t.Run("return fallback creds if Barbrady returns something different for developer", func(t *testing.T) {
		allPermissionsCartmanCreds := AllPermissions()
		params := map[string]string{ParamExtensionID: fakeExtensionId}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(allPermissionsCartmanCreds, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapEditDeveloperConfig}, params)

		barbradyCredsGranted := barbradyCredentials.BarbradyCanEditConfig(fakeExtensionId, developerSegment)
		fallbackGranted := allPermissionsCartmanCreds.CanEditConfig(fakeExtensionId, developerSegment)
		credsGranted := barbradyCredentials.CanEditConfig(fakeExtensionId, developerSegment)
		assert.False(t, barbradyCredsGranted)
		assert.True(t, fallbackGranted)
		assert.NotEqual(t, credsGranted, barbradyCredsGranted)
		assert.Equal(t, credsGranted, fallbackGranted)
	})
}

// CanDeleteConfig is granted as mock_success, Barbrady *should* normally return true
func runBarbradyCanReadConfigTests(t *testing.T, fakeCartmanCredentials *FakeCredentials, fakeBarbradyClient *fakes.FakeTwitchExtensionsBarbrady, mockStatsd statsdClient.Statter, segment protocol.Segment) {
	params := map[string]string{}

	t.Run("return true if user is granted capability with matching params", func(t *testing.T) {
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{
				{
					Name:       CapReadConfig,
					Granted:    true,
					Parameters: params,
				},
			},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapReadConfig}, params)
		granted := barbradyCredentials.BarbradyCanReadConfig(fakeExtensionId, segment)
		assert.True(t, granted)
	})

	t.Run("return false if user was not granted capability", func(t *testing.T) {
		params := map[string]string{}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{
				{
					Name:       CapReadConfig,
					Granted:    false,
					Parameters: params,
				},
			},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapReadConfig}, params)
		granted := barbradyCredentials.BarbradyCanReadConfig(fakeExtensionId, segment)
		assert.False(t, granted)
	})

	t.Run("return false if capability not returned from barbrady", func(t *testing.T) {
		params := map[string]string{}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(fakeCartmanCredentials, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapReadConfig}, params)
		granted := barbradyCredentials.BarbradyCanReadConfig(fakeExtensionId, segment)
		assert.False(t, granted)
	})

	t.Run("return fallback creds if Barbrady returns something different", func(t *testing.T) {
		allPermissionsCartmanCreds := AllPermissions()
		params := map[string]string{}
		fakeBarbradyClient.GetCapabilitiesReturns(&barbradytwirp.GetCapabilitiesResponse{
			Capabilities: []*barbradytwirp.Capability{},
		}, nil)

		barbradyCredentials := NewBarbradyCredentials(allPermissionsCartmanCreds, fakeBarbradyClient, mockStatsd).(*BarbradyCredentials)
		barbradyCredentials.RequestCapabilities([]string{CapReadConfig}, params)

		barbradyCredsGranted := barbradyCredentials.BarbradyCanReadConfig(fakeExtensionId, segment)
		fallbackGranted := allPermissionsCartmanCreds.CanReadConfig(fakeExtensionId, segment)
		credsGranted := barbradyCredentials.CanReadConfig(fakeExtensionId, segment)
		assert.False(t, barbradyCredsGranted)
		assert.True(t, fallbackGranted)
		assert.NotEqual(t, credsGranted, barbradyCredsGranted)
		assert.Equal(t, credsGranted, fallbackGranted)
	})
}
