package auth

import (
	"fmt"
	"testing"
	"time"

	ga "code.justin.tv/common/goauthorization"
	"code.justin.tv/common/jwt/claim"
	"code.justin.tv/extensions/configuration/services/main/protocol"
	"github.com/stretchr/testify/assert"
)

const (
	testAudience = "test-audience"
	testIssuer   = "test-issuer"
)

var (
	extID          = "extID"
	chID           = "chID"
	global         = protocol.Global()
	broadcaster, _ = protocol.Broadcaster(chID)
	developer, _   = protocol.Developer(chID)
	testDecoder, _ = ga.NewDecoder("HS512", testAudience, testIssuer, []byte("unused"))
)

func createToken(userID string, clientID string) *ga.AuthorizationToken {
	token := new(ga.AuthorizationToken)
	token.Claims.Expires = claim.Exp(time.Now().Add(time.Minute))
	token.Claims.Sub.Sub = userID
	token.Claims.ClientID = clientID
	token.Claims.Audience = []string{testAudience}
	token.Claims.Issuer = claim.Iss(testIssuer)
	return token
}

func TestCartmanCredentials_defaults(t *testing.T) {
	t.Run("NewCartmanCredentials", func(t *testing.T) {
		t.Run("grants read by default", func(t *testing.T) {
			token := createToken("", "")
			creds := NewCartmanCredentials(testDecoder, token)
			assert.Nil(t, creds.UserID())
			assert.True(t, creds.CanReadConfig(extID, global))
			assert.False(t, creds.CanEditConfig(extID, global))
		})
	})
}

func TestCartmanCredentials_UserID(t *testing.T) {
	t.Run("returns nil if no user is authenticated", func(t *testing.T) {
		token := createToken("", "")
		creds := NewCartmanCredentials(testDecoder, token)
		assert.Nil(t, creds.UserID())
	})

	t.Run("returns the authenticated user id if present", func(t *testing.T) {
		token := createToken("1234", "")
		creds := NewCartmanCredentials(testDecoder, token)
		assert.Equal(t, &token.Claims.Sub.Sub, creds.UserID())
	})
}

func TestCartmanCredentials_CanDeleteConfig(t *testing.T) {
	t.Run("returns false if no permission is present", func(t *testing.T) {
		token := createToken("", "")
		creds := NewCartmanCredentials(testDecoder, token)
		assert.False(t, creds.CanDeleteConfig(chID))
	})

	t.Run("returns true for the broadcaster", func(t *testing.T) {
		token := createToken("", "")
		token.Claims.Authorizations = ga.CapabilityClaims{
			CapEditBroadcasterConfig: ga.CapabilityClaim{ParamChannelID: chID},
		}
		creds := NewCartmanCredentials(testDecoder, token)
		assert.True(t, creds.CanDeleteConfig(chID))
	})

	t.Run("returns false for other broadcasters", func(t *testing.T) {
		token := createToken("", "")
		token.Claims.Authorizations = ga.CapabilityClaims{
			CapEditBroadcasterConfig: ga.CapabilityClaim{ParamChannelID: "other"},
		}
		creds := NewCartmanCredentials(testDecoder, token)
		assert.False(t, creds.CanDeleteConfig(chID))
	})

	t.Run("returns false for developers", func(t *testing.T) {
		token := createToken("", "")
		token.Claims.Authorizations = ga.CapabilityClaims{
			CapEditDeveloperConfig: ga.CapabilityClaim{ParamExtensionID: extID},
		}
		creds := NewCartmanCredentials(testDecoder, token)
		assert.False(t, creds.CanDeleteConfig(chID))
	})
}

func TestCartmanCredentials_CanEditConfig(t *testing.T) {
	t.Run("for broadcaster segments", func(t *testing.T) {
		t.Run("returns false if no permission is present", func(t *testing.T) {
			token := createToken("", "")
			creds := NewCartmanCredentials(testDecoder, token)
			assert.False(t, creds.CanEditConfig(extID, broadcaster))
		})

		t.Run("returns true for the broadcaster", func(t *testing.T) {
			token := createToken("", "")
			token.Claims.Authorizations = ga.CapabilityClaims{
				CapEditBroadcasterConfig: ga.CapabilityClaim{ParamChannelID: chID},
			}
			creds := NewCartmanCredentials(testDecoder, token)
			assert.True(t, creds.CanEditConfig(extID, broadcaster))
		})

		t.Run("returns false for other broadcasters", func(t *testing.T) {
			token := createToken("", "")
			token.Claims.Authorizations = ga.CapabilityClaims{
				CapEditBroadcasterConfig: ga.CapabilityClaim{ParamChannelID: "other"},
			}
			creds := NewCartmanCredentials(testDecoder, token)
			assert.False(t, creds.CanEditConfig(extID, broadcaster))
		})

		t.Run("returns true for the developer", func(t *testing.T) {
			token := createToken("", "")
			token.Claims.Authorizations = ga.CapabilityClaims{
				CapEditDeveloperConfig: ga.CapabilityClaim{ParamExtensionID: extID},
			}
			creds := NewCartmanCredentials(testDecoder, token)
			assert.True(t, creds.CanEditConfig(extID, broadcaster))
		})

		t.Run("returns false for other developers", func(t *testing.T) {
			token := createToken("", "")
			token.Claims.Authorizations = ga.CapabilityClaims{
				CapEditDeveloperConfig: ga.CapabilityClaim{ParamExtensionID: "other"},
			}
			creds := NewCartmanCredentials(testDecoder, token)
			assert.False(t, creds.CanEditConfig(extID, broadcaster))
		})
	})

	t.Run("for developer segments", func(t *testing.T) {
		t.Run("returns false if no permission is present", func(t *testing.T) {
			token := createToken("", "")
			creds := NewCartmanCredentials(testDecoder, token)
			assert.False(t, creds.CanEditConfig(extID, developer))
		})

		t.Run("returns false for the broadcaster", func(t *testing.T) {
			token := createToken("", "")
			token.Claims.Authorizations = ga.CapabilityClaims{
				CapEditBroadcasterConfig: ga.CapabilityClaim{ParamChannelID: chID},
			}
			creds := NewCartmanCredentials(testDecoder, token)
			assert.False(t, creds.CanEditConfig(extID, developer))
		})

		t.Run("returns true for the developer", func(t *testing.T) {
			token := createToken("", "")
			token.Claims.Authorizations = ga.CapabilityClaims{
				CapEditDeveloperConfig: ga.CapabilityClaim{ParamExtensionID: extID},
			}
			creds := NewCartmanCredentials(testDecoder, token)
			assert.True(t, creds.CanEditConfig(extID, developer))
		})

		t.Run("returns false for other developers", func(t *testing.T) {
			token := createToken("", "")
			token.Claims.Authorizations = ga.CapabilityClaims{
				CapEditDeveloperConfig: ga.CapabilityClaim{ParamExtensionID: "other"},
			}
			creds := NewCartmanCredentials(testDecoder, token)
			assert.False(t, creds.CanEditConfig(extID, developer))
		})
	})

	t.Run("for global segments", func(t *testing.T) {
		t.Run("returns false if no permission is present", func(t *testing.T) {
			token := createToken("", "")
			creds := NewCartmanCredentials(testDecoder, token)
			assert.False(t, creds.CanEditConfig(extID, global))
		})

		t.Run("returns false for broadcasters", func(t *testing.T) {
			token := createToken("", "")
			token.Claims.Authorizations = ga.CapabilityClaims{
				CapEditBroadcasterConfig: ga.CapabilityClaim{ParamChannelID: chID},
			}
			creds := NewCartmanCredentials(testDecoder, token)
			assert.False(t, creds.CanEditConfig(extID, global))
		})

		t.Run("returns true for the developer", func(t *testing.T) {
			token := createToken("", "")
			token.Claims.Authorizations = ga.CapabilityClaims{
				CapEditDeveloperConfig: ga.CapabilityClaim{ParamExtensionID: extID},
			}
			creds := NewCartmanCredentials(testDecoder, token)
			assert.True(t, creds.CanEditConfig(extID, global))
		})

		t.Run("returns false for other developers", func(t *testing.T) {
			token := createToken("", "")
			token.Claims.Authorizations = ga.CapabilityClaims{
				CapEditDeveloperConfig: ga.CapabilityClaim{ParamExtensionID: "other"},
			}
			creds := NewCartmanCredentials(testDecoder, token)
			assert.False(t, creds.CanEditConfig(extID, global))
		})
	})
}

func TestCartmanCredentials_CanReadConfig(t *testing.T) {
	t.Run("returns true for any valid token", func(t *testing.T) {
		token := createToken("", "")
		creds := NewCartmanCredentials(testDecoder, token)
		assert.True(t, creds.CanReadConfig(extID, global))
		assert.True(t, creds.CanReadConfig(extID, developer))
		assert.True(t, creds.CanReadConfig(extID, broadcaster))
	})
}

func TestCartmanCredentials_String(t *testing.T) {
	token := createToken("", "")
	creds := NewCartmanCredentials(testDecoder, token)
	assert.Equal(t, fmt.Sprintf("%+v", token.Claims), fmt.Sprintf("%v", creds))
}
