package auth

import (
	"testing"

	"code.justin.tv/extensions/configuration/services/main/protocol"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestFakeCredentials(t *testing.T) {
	ch := "chID"
	ch2 := "ch2ID"
	dev, err := protocol.Developer(ch)
	require.NoError(t, err)
	dev2, err := protocol.Developer(ch2)
	require.NoError(t, err)
	broad, err := protocol.Broadcaster(ch)
	require.NoError(t, err)
	broad2, err := protocol.Broadcaster(ch2)
	require.NoError(t, err)

	ext := "ext1ID"
	ext2 := "ext2ID"
	other := "ext3ID"

	t.Run("grants read by default", func(t *testing.T) {
		creds := &FakeCredentials{}
		assert.Nil(t, creds.UserID())
		assert.Equal(t, "", creds.ClientID())
		assert.Equal(t, "FakeCredentials(,,[])", creds.String())
		assert.False(t, creds.CanDeleteConfig(ch))
		assert.False(t, creds.CanDeleteConfig(""))
		assert.False(t, creds.CanEditConfig(ext, protocol.Global()))
		assert.False(t, creds.CanEditConfig(ext, dev))
		assert.False(t, creds.CanEditConfig(ext, broad))
		assert.True(t, creds.CanReadConfig(ext, protocol.Global()))
		assert.True(t, creds.CanReadConfig(ext, dev))
		assert.True(t, creds.CanReadConfig(ext, broad))
	})

	t.Run("reports userID correctly", func(t *testing.T) {
		userID := "1"
		creds := &FakeCredentials{UserIDValue: userID}
		require.NotNil(t, creds.UserID())
		assert.Equal(t, userID, *creds.UserID())
		assert.Equal(t, "", creds.ClientID())
		assert.Equal(t, "FakeCredentials(1,,[])", creds.String())
		assert.False(t, creds.CanDeleteConfig(userID))
		assert.False(t, creds.CanEditConfig(ext, protocol.Global()))
		assert.False(t, creds.CanEditConfig(ext, dev))
		assert.False(t, creds.CanEditConfig(ext, broad))
		assert.True(t, creds.CanReadConfig(ext, protocol.Global()))
		assert.True(t, creds.CanReadConfig(ext, dev))
		assert.True(t, creds.CanReadConfig(ext, broad))
	})

	t.Run("supports channel ownership correctly", func(t *testing.T) {
		t.Run("for specific channel ownership", func(t *testing.T) {
			creds := &FakeCredentials{BroadcasterChannel: ch}
			assert.True(t, creds.CanDeleteConfig(ch))
			assert.False(t, creds.CanEditConfig(ext, protocol.Global()))
			assert.False(t, creds.CanEditConfig(ext, dev))
			assert.True(t, creds.CanEditConfig(ext, broad))
			assert.True(t, creds.CanReadConfig(ext, protocol.Global()))
			assert.True(t, creds.CanReadConfig(ext, dev))
			assert.True(t, creds.CanReadConfig(ext, broad))

			assert.False(t, creds.CanDeleteConfig(ch2))
			assert.False(t, creds.CanEditConfig(ext, protocol.Global()))
			assert.False(t, creds.CanEditConfig(ext, dev2))
			assert.False(t, creds.CanEditConfig(ext, broad2))
			assert.True(t, creds.CanReadConfig(ext, protocol.Global()))
			assert.True(t, creds.CanReadConfig(ext, dev2))
			assert.True(t, creds.CanReadConfig(ext, broad2))
		})

		t.Run("for wildcard channel ownership", func(t *testing.T) {
			creds := &FakeCredentials{BroadcasterChannel: "*"}
			assert.True(t, creds.CanDeleteConfig(ch))
			assert.False(t, creds.CanEditConfig(ext, protocol.Global()))
			assert.False(t, creds.CanEditConfig(ext, dev))
			assert.True(t, creds.CanEditConfig(ext, broad))
			assert.True(t, creds.CanReadConfig(ext, protocol.Global()))
			assert.True(t, creds.CanReadConfig(ext, dev))
			assert.True(t, creds.CanReadConfig(ext, broad))

			assert.True(t, creds.CanDeleteConfig(ch2))
			assert.False(t, creds.CanEditConfig(ext, protocol.Global()))
			assert.False(t, creds.CanEditConfig(ext, dev2))
			assert.True(t, creds.CanEditConfig(ext, broad2))
			assert.True(t, creds.CanReadConfig(ext, protocol.Global()))
			assert.True(t, creds.CanReadConfig(ext, dev2))
			assert.True(t, creds.CanReadConfig(ext, broad2))
		})
	})

	t.Run("handles extension ownership correctly", func(t *testing.T) {
		t.Run("for specific extension ownership", func(t *testing.T) {
			creds := &FakeCredentials{OwnedExtensions: []string{ext, ext2}}
			assert.False(t, creds.CanDeleteConfig(ch))

			assert.True(t, creds.CanEditConfig(ext, protocol.Global()))
			assert.True(t, creds.CanEditConfig(ext, dev))
			assert.True(t, creds.CanEditConfig(ext, broad))
			assert.True(t, creds.CanReadConfig(ext, protocol.Global()))
			assert.True(t, creds.CanReadConfig(ext, dev))
			assert.True(t, creds.CanReadConfig(ext, broad))

			assert.True(t, creds.CanEditConfig(ext2, protocol.Global()))
			assert.True(t, creds.CanEditConfig(ext2, dev))
			assert.True(t, creds.CanEditConfig(ext2, broad))
			assert.True(t, creds.CanReadConfig(ext2, protocol.Global()))
			assert.True(t, creds.CanReadConfig(ext2, dev))
			assert.True(t, creds.CanReadConfig(ext2, broad))

			assert.False(t, creds.CanEditConfig(other, protocol.Global()))
			assert.False(t, creds.CanEditConfig(other, dev))
			assert.False(t, creds.CanEditConfig(other, broad))
			assert.True(t, creds.CanReadConfig(other, protocol.Global()))
			assert.True(t, creds.CanReadConfig(other, dev))
			assert.True(t, creds.CanReadConfig(other, broad))
		})

		t.Run("for wildcard extension ownership", func(t *testing.T) {
			creds := &FakeCredentials{OwnedExtensions: []string{"*"}}
			assert.False(t, creds.CanDeleteConfig(ch))

			assert.True(t, creds.CanEditConfig(ext, protocol.Global()))
			assert.True(t, creds.CanEditConfig(ext, dev))
			assert.True(t, creds.CanEditConfig(ext, broad))
			assert.True(t, creds.CanReadConfig(ext, protocol.Global()))
			assert.True(t, creds.CanReadConfig(ext, dev))
			assert.True(t, creds.CanReadConfig(ext, broad))

			assert.True(t, creds.CanEditConfig(ext2, protocol.Global()))
			assert.True(t, creds.CanEditConfig(ext2, dev))
			assert.True(t, creds.CanEditConfig(ext2, broad))
			assert.True(t, creds.CanReadConfig(ext2, protocol.Global()))
			assert.True(t, creds.CanReadConfig(ext2, dev))
			assert.True(t, creds.CanReadConfig(ext2, broad))

			assert.True(t, creds.CanEditConfig(other, protocol.Global()))
			assert.True(t, creds.CanEditConfig(other, dev))
			assert.True(t, creds.CanEditConfig(other, broad))
			assert.True(t, creds.CanReadConfig(other, protocol.Global()))
			assert.True(t, creds.CanReadConfig(other, dev))
			assert.True(t, creds.CanReadConfig(other, broad))
		})
	})
}

func TestAllPermissions(t *testing.T) {
	t.Run("should have all powers", func(t *testing.T) {
		dev, err := protocol.Developer("chID")
		require.NoError(t, err)

		creds := AllPermissions()
		assert.True(t, creds.CanDeleteConfig("chID"))
		assert.True(t, creds.CanEditConfig("extID", protocol.Global()))
		assert.True(t, creds.CanEditConfig("extID", dev))
		assert.True(t, creds.CanEditConfig("extID2", protocol.Global()))
		assert.True(t, creds.CanEditConfig("extID2", dev))
		assert.True(t, creds.CanReadConfig("extID", protocol.Global()))
		assert.True(t, creds.CanReadConfig("extID", dev))
		assert.True(t, creds.CanReadConfig("extID2", protocol.Global()))
		assert.True(t, creds.CanReadConfig("extID2", dev))
		assert.Nil(t, creds.UserID())
	})
}
