package main

import (
	"code.justin.tv/foundation/twitchclient"
	"fmt"
	"log"
	"net/http"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/client"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/gds/gds/golibs/config"
	"code.justin.tv/gds/gds/golibs/config/builders"
	"code.justin.tv/gds/gds/golibs/config/sources"
	"code.justin.tv/gds/gds/golibs/event"
	"code.justin.tv/gds/gds/golibs/sandstorm"
	"code.justin.tv/gds/gds/golibs/uuid"

	configuration "code.justin.tv/extensions/configuration/services/main"
	"code.justin.tv/extensions/configuration/services/main/api"
	"code.justin.tv/extensions/configuration/services/main/auth"
	"code.justin.tv/extensions/configuration/services/main/data/controller"
	"code.justin.tv/extensions/configuration/services/main/data/model"
	"code.justin.tv/extensions/configuration/services/main/data/model/cached"
	"code.justin.tv/extensions/configuration/services/main/data/model/dynamo"
	"code.justin.tv/extensions/configuration/services/main/data/model/memory"
	"code.justin.tv/extensions/configuration/services/main/listeners"

	barbradyTwirp "code.justin.tv/amzn/TwitchExtensionsBarbradyTwirp"
	fultonConfigService "code.justin.tv/extensions/fulton-configuration/client"
)

const (
	localDebugPath        = "config/local_debug.config.json"
	defaultHystrixTimeout = 3000

	configAppEnvironment           = "app.env"
	configAppComponentName         = "app.component_name"
	configAppName                  = "app.name"
	configAuthMethod               = "auth.method"
	configBarbradySamplingRate     = "auth.barbrady.sampling.rate"
	configCacheDuration            = "store.cache.duration"
	configCartmanSecretName        = "auth.cartman.secret_name"
	configFultonRoutingPercentBeta = "app.fulton_routing_percent_beta"
	configFultonRoutingPercentProd = "app.fulton_routing_percent_prod"
	configFultonHostBeta           = "app.config_fulton_host_beta"
	configFultonHostProd           = "app.config_fulton_host_prod"
	configSandstormRegion          = "auth.sandstorm.region"
	configSandstormRoleARN         = "auth.sandstorm.role_arn"
	configSandstormTable           = "auth.sandstorm.table"
	configSandstormKeyID           = "auth.sandstorm.key_id"
	configEventBusARN              = "events.sns.arn"
	configStatsMethod              = "stats.method"
	configStatsCommon              = configuration.ConfigStatsCommon
	configStatsFrequent            = configuration.ConfigStatsFrequent
	configStatsdHost               = "stats.statsd.host"
	configStatsRare                = configuration.ConfigStatsRare
	configDynamoAggregationTime    = "store.dynamo.aggregation_time"
	configDynamoPrefix             = "store.dynamo.prefix"
	configStoreMethod              = "store.method"
)

func loadEnv(cfg config.Config, cp client.ConfigProvider) sources.RefreshableSource {
	src, err := sources.NewRefreshableSource(cfg, sources.NewEnvironmentRefreshLogic([]string{
		configAppEnvironment,
		configAppComponentName,
		configAppName,
		configAuthMethod,
		configCartmanSecretName,
		configDynamoPrefix,
		configEventBusARN,
		configSandstormRegion,
		configSandstormRoleARN,
		configSandstormTable,
		configSandstormKeyID,
		configStatsCommon,
		configStatsdHost,
		configStatsFrequent,
		configStatsMethod,
		configStatsRare,
		configDynamoAggregationTime,
		configDynamoPrefix,
		configStoreMethod,
		configBarbradySamplingRate,
	}))
	if err != nil {
		panic(fmt.Errorf("Unable to initialize env: %v", err))
	}
	return src
}

func createConfig(cp client.ConfigProvider) (config.Config, config.RefreshController) {
	defaults := builders.StandardDefaults{
		ComponentName: "configuration",
		Env:           "dev",
		Name:          "ext",
		Sources: []*builders.Source{
			{Type: builders.File, Refresh: 10 * time.Second},
			{
				Type:     builders.Custom,
				Refresh:  1 * time.Second,
				Callback: loadEnv,
				Name:     "env",
			},
		},
		AdditionalDefaults: map[string]interface{}{
			configAuthMethod:               "cartman",
			configCacheDuration:            "100ms",
			configStatsMethod:              "statsd",
			configStatsdHost:               "statsd.internal.justin.tv:8125",
			configStoreMethod:              "dynamo",
			configDynamoAggregationTime:    "1ms",
			configDynamoPrefix:             "configuration",
			configBarbradySamplingRate:     0,
			configFultonRoutingPercentBeta: 100,
			configFultonRoutingPercentProd: 0,
			configFultonHostBeta:           "https://us-west-2.beta.twitchextensionsconfiguration.s.twitch.a2z.com",
			configFultonHostProd:           "https://us-west-2.prod.twitchextensionsconfiguration.s.twitch.a2z.com",
		},
	}
	cfg, ctrl := defaults.CreateConfigAndSchedule(cp)
	log.Printf("\nConfiguration:\n%v", cfg.List().PrintTable())
	return cfg, ctrl
}

func createEventCoordinator(cp client.ConfigProvider, cfg config.Config) *event.Coordinator {
	coord := event.NewCoordinator(true)
	if arn, ok := cfg.TryGetString(configEventBusARN); ok && arn != "" {
		coord.Register(listeners.NewEventBusListener(sns.New(cp), cfg.RequireString(configAppEnvironment), arn))
	}
	return coord
}

func createMetrics(cfg config.Config) statsd.Statter {
	switch cfg.RequireString(configStatsMethod) {
	case "noop":
		statter, _ := statsd.NewNoopClient()
		return statter
	case "statsd":
		prefix := fmt.Sprintf("%s.%s", cfg.RequireString(configAppComponentName), cfg.RequireString(configAppEnvironment))
		statter, err := statsd.NewBufferedClient(cfg.RequireString(configStatsdHost), prefix, time.Second, 0)

		if err != nil {
			panic(fmt.Errorf("Error creating statsd client: %v", err))
		}
		return statter
	default:
		panic(fmt.Errorf("Unknown stat method: %v", cfg.RequireString(configStatsMethod)))
	}
}

func createDataCache(cfg config.Config, store model.Store) model.Store {
	duration, err := time.ParseDuration(cfg.RequireString(configCacheDuration))
	if err != nil {
		panic(fmt.Errorf("Illegal '%s' specified: %v", configCacheDuration, err))
	}
	// TODO : support dynamic cache duration tuning -- this requires updating golibs
	return cached.New(store, duration)
}

func createDataStore(cp client.ConfigProvider, cfg config.Config) model.Store {
	switch cfg.RequireString(configStoreMethod) {
	case "memory":
		return memory.New(uuid.NewSource())
	case "dynamo":
		aggr, err := time.ParseDuration(cfg.RequireString(configDynamoAggregationTime))
		if err != nil {
			panic(fmt.Errorf("Illegal '%s' specified: %v", configDynamoAggregationTime, err))
		}
		return dynamo.New(uuid.NewSource(), dynamodb.New(cp), nil, cfg.RequireString(configDynamoPrefix), aggr)
	default:
		panic(fmt.Errorf("Unknown storage method: %v", cfg.RequireString(configStoreMethod)))
	}
}

func createAuthHandler(cfg config.Config, statter statsd.Statter) auth.Handler {
	switch cfg.RequireString(configAuthMethod) {
	case "fake":
		// TODO : revisit ability to do this outside of dev environments.
		log.Printf("Installing a fake auth handler; should allow everything")
		return &auth.FakeHandler{auth.AllPermissions()}
	case "cartman":
		log.Printf("Installing JWT/Cartman authentication")
		decoder, err := sandstorm.NewDecoder(
			cfg.RequireString(configSandstormRegion),
			cfg.RequireString(configSandstormRoleARN),
			cfg.RequireString(configSandstormTable),
			cfg.RequireString(configSandstormKeyID),
			cfg.RequireString(configCartmanSecretName),
			auth.CartmanAudience,
			auth.CartmanIssuer)
		if err != nil {
			panic(fmt.Errorf("Unable to initialize Sandstorm: %v", err))
		}

		var barbradyHostName string
		switch cfg.RequireString(configAppEnvironment) {
		case "dev":
			barbradyHostName = "https://us-west-2.beta.twitchextensionsbarbrady.s.twitch.a2z.com"
		case "prod":
			barbradyHostName = "https://us-west-2.prod.twitchextensionsbarbrady.s.twitch.a2z.com"
		}

		barbradyClient := createBarbradyClient(barbradyHostName)
		barbradySamplingPercent := cfg.RequireInt(configBarbradySamplingRate)
		return auth.NewJWTHandler(decoder, barbradyClient, barbradySamplingPercent, statter)
	default:
		panic(fmt.Errorf("Unknown auth method: %v", cfg.RequireString(configAuthMethod)))
	}
}

func createBarbradyClient(barbradyHostName string) barbradyTwirp.TwitchExtensionsBarbrady {
	return barbradyTwirp.NewTwitchExtensionsBarbradyProtobufClient(
		barbradyHostName,
		&http.Client{},
	)
}

func createFultonConfigServiceClient(cfg config.Config, statter statsd.Statter) (fultonConfigService.Client, error) {
	var configServiceHost string
	switch cfg.RequireString(configAppEnvironment) {
	case "dev":
		configServiceHost = cfg.RequireString(configFultonHostBeta)
	case "prod":
		configServiceHost = cfg.RequireString(configFultonHostProd)
	}

	configurationConf := twitchclient.ClientConf{
		Transport: twitchclient.TransportConf{
			MaxIdleConnsPerHost: 200,
		},
		RoundTripperWrappers: nil,
		Stats:                statter,
		Host:                 configServiceHost,
	}

	return fultonConfigService.NewClient(configurationConf)
}

func main() {
	awsConfig := aws.NewConfig().WithRegion("us-west-2").WithCredentialsChainVerboseErrors(true)

	awsSession, err := session.NewSession(awsConfig)
	if err != nil {
		panic(err)
	}

	cfg, ctrl := createConfig(awsSession)
	defer ctrl.UnscheduleAll()

	statter := createMetrics(cfg)
	defer statter.Close()

	handler := createAuthHandler(cfg, statter)
	store := createDataStore(awsSession, cfg)
	store = createDataCache(cfg, store)
	coord := createEventCoordinator(awsSession, cfg)
	mgr := controller.New(store, coord)

	fultonConfigService, err := createFultonConfigServiceClient(cfg, statter)
	if err != nil {
		panic(fmt.Sprintf("Could not create client for Fulton Config Service: %v", err))
	}

	var routingPercent int64
	switch cfg.RequireString(configAppEnvironment) {
	case "dev":
		routingPercent = cfg.RequireInt(configFultonRoutingPercentBeta)
	case "prod":
		routingPercent = cfg.RequireInt(configFultonRoutingPercentProd)
	}

	api := api.NewAPI(routingPercent, fultonConfigService)

	twitchserver.AddDefaultSignalHandlers()
	server := configuration.BuildServer(api, statter, defaultHystrixTimeout, cfg, handler, mgr)
	err = twitchserver.ListenAndServe(server, nil)
	if err != nil {
		err = fmt.Errorf("ListenAndServe failed with: %s", err)
		log.Print(err)
	}
}
