package data

import (
	"context"
	"net/http"
	"testing"

	goji "goji.io"
	"goji.io/pat"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

type mockWriter struct {
	http.ResponseWriter
}

type mockManager struct {
	ConfigurationManager
}

func (m *mockManager) WithContext(ctx context.Context) ConfigurationManager {
	return m
}

func TestLoadMissingManager(t *testing.T) {
	m := Load(context.Background())
	assert.NotNil(t, m)
	assert.Equal(t, &nullManager{}, m)
}

func TestLoadCreatedConfig(t *testing.T) {
	var m ConfigurationManager = &mockManager{}
	ctx := context.WithValue(context.Background(), &managerKey, m)
	out := Load(ctx)
	assert.NotNil(t, m)
	assert.Equal(t, m, out)
	assert.NotEqual(t, &nullManager{}, out)
}

func TestStore(t *testing.T) {
	m := &nullManager{}
	ctx := Store(context.Background(), m)
	assert.Equal(t, m, Load(ctx))
}

func TestStoreManager(t *testing.T) {
	var m ConfigurationManager = &nullManager{}
	called := false
	inner := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		assert.Equal(t, m, Load(r.Context()))
		called = true
	})
	server := goji.NewMux()
	server.Use(StoreManager(m))
	server.Handle(pat.Get("/"), inner)

	req, err := http.NewRequest("GET", "/", nil)
	require.NoError(t, err)
	server.ServeHTTP(new(mockWriter), req)
	assert.True(t, called)
}
