package cached

import (
	"strings"
	"time"

	"code.justin.tv/extensions/configuration/services/main/data/model"
	"code.justin.tv/extensions/configuration/services/main/protocol"
	"code.justin.tv/gds/gds/golibs/cache"
)

type store struct {
	cache  *cache.Local
	source model.Store
}

func New(source model.Store, refreshAfter time.Duration) model.Store {
	store := &store{source: source}
	store.cache = cache.NewAsync(store.getForCache, refreshAfter)
	return store
}

func (s *store) AsyncLoadChannel(env, extID, chID string) model.ChannelPromise {
	if cast, ok := s.cache.Load(channelKey(env, extID, chID)).(model.ChannelPromise); ok {
		return cast
	}
	return model.NewErrorPromise(protocol.ErrInvalidCacheContent)
}

func (s *store) LoadChannel(env, extID, chID string) (*model.Channel, error) {
	return s.AsyncLoadChannel(env, extID, chID).Channel()
}

func (s *store) AsyncLoadCommon(env, extID string) model.CommonPromise {
	if cast, ok := s.cache.Load(commonKey(env, extID)).(model.CommonPromise); ok {
		return cast
	}
	return model.NewErrorPromise(protocol.ErrInvalidCacheContent)
}

func (s *store) LoadCommon(env, extID string) (*model.Common, error) {
	return s.AsyncLoadCommon(env, extID).Common()
}

func (s *store) SaveCommon(c *model.Common) error            { return s.source.SaveCommon(c) }
func (s *store) SaveChannel(c *model.Channel) error          { return s.source.SaveChannel(c) }
func (s *store) DeleteChannel(chID string) error             { return s.source.DeleteChannel(chID) }
func (s *store) MarkCommonPublished(c *model.Common) error   { return s.source.MarkCommonPublished(c) }
func (s *store) MarkChannelPublished(c *model.Channel) error { return s.source.MarkChannelPublished(c) }
func (s *store) IsResetEnabled() bool                        { return s.source.IsResetEnabled() }
func (s *store) EnableDataReset()                            { s.source.EnableDataReset() }
func (s *store) ResetAllData() error {
	if err := s.source.ResetAllData(); err != nil {
		return err
	}
	s.cache.Reset()
	return nil
}

func (s *store) getForCache(key string) cache.Promise {
	segs := strings.Split(key, ":")
	switch segs[0] {
	case "c":
		switch len(segs) {
		case 3:
			return s.source.AsyncLoadCommon(segs[1], segs[2])
		case 4:
			return s.source.AsyncLoadChannel(segs[1], segs[2], segs[3])
		}
	}
	return model.NewErrorPromise(protocol.ErrInvalidCacheKey)
}

func commonKey(env, extID string) string        { return "c:" + env + ":" + extID }
func channelKey(env, extID, chID string) string { return "c:" + env + ":" + extID + ":" + chID }
