package model

import (
	"encoding/json"
	"errors"
	"testing"
	"time"

	"code.justin.tv/extensions/configuration/services/main/protocol"
	"code.justin.tv/extensions/configuration/services/main/protocol/messages"
	"code.justin.tv/gds/gds/golibs/dynamodb/lazy"
	"code.justin.tv/gds/gds/golibs/event"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

type badMessage struct{}

func (badMessage) FormatVersion() int           { return 0 }
func (badMessage) Topic() event.Topic           { return event.Topic("x") }
func (badMessage) Change() event.Change         { return event.Change("y") }
func (b badMessage) Body() interface{}          { return b }
func (badMessage) MarshalJSON() ([]byte, error) { return nil, errors.New("expected") }

func TestChannel_Memento(t *testing.T) {
	start, err := time.Parse(time.RFC3339, "1999-12-31T23:59:59Z")
	require.NoError(t, err)

	message := messages.NewConfigMessage(1, messages.OnSet)
	message.Time = start

	t.Run("should copy new values", func(t *testing.T) {
		ch := NewChannel("", "", "")
		m, err := ch.GetMemento()
		require.NoError(t, err)

		var ch2 Channel
		assert.NoError(t, ch2.ApplyMemento(m))
		assert.Equal(t, *ch, ch2)
	})

	t.Run("should preserve values", func(t *testing.T) {
		ch := NewChannel("env", "extID", "chID")

		content := "value"
		ch.Broadcaster = protocol.NewRecord("1", &content)
		other := "other"
		ch.Developer = protocol.NewRecord("2", &other)
		ch.ConcurrencyUUID = "1234"
		ch.MsgSeq = -42
		ch.Messages = []event.Message{message}
		ch.UnpublishedTime = &start
		m, err := ch.GetMemento()
		require.NoError(t, err)

		var ch2 Channel
		assert.NoError(t, ch2.ApplyMemento(m))
		assert.Equal(t, *ch, ch2)
	})

	t.Run("should error when given the wrong type", func(t *testing.T) {
		var ch Channel
		assert.Equal(t, lazy.ErrMigrationCorrupt, ch.ApplyMemento("broken"))
	})

	t.Run("should forward bad message errors on get", func(t *testing.T) {
		ch := NewChannel("env", "extID", "chID")
		ch.Messages = []event.Message{&badMessage{}}
		m, err := ch.GetMemento()
		assert.Nil(t, m)
		assert.IsType(t, &json.MarshalerError{}, err)
	})

	t.Run("should forward bad message errors on apply", func(t *testing.T) {
		ch := NewChannel("env", "extID", "chID")
		m, err := ch.GetMemento()
		require.NoError(t, err)
		m.(*channelMemento).Messages = []string{"{"}
		var ch2 Channel
		assert.Equal(t, event.ErrMalformedMessage, ch2.ApplyMemento(m))
	})
}

func TestChannel_Migration(t *testing.T) {
	start, err := time.Parse(time.RFC3339, "1999-12-31T23:59:59Z")
	require.NoError(t, err)

	message := messages.NewConfigMessage(1, messages.OnSet)
	message.Time = start
	content := "value"
	src := &Channel{
		Environment:     "env",
		ExtensionID:     "extID",
		ChannelID:       "chID",
		Broadcaster:     protocol.NewRecord("1", nil),
		Developer:       protocol.NewRecord("2", &content),
		Messages:        []event.Message{message},
		MsgSeq:          1,
		UnpublishedTime: &start,
	}

	av, err := lazy.Marshal(src)
	assert.NoError(t, err)
	require.NotNil(t, av)

	dest := &Channel{}
	assert.NoError(t, lazy.Unmarshal(av, dest))
	assert.EqualValues(t, src, dest)
}
