package dynamo

import (
	"time"

	"code.justin.tv/extensions/configuration/services/main/data/model"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

// note : these functions are part of *store, but placed here to keep file sizes
// sane
func (s *store) IsBlocked(channelID string) model.BlockPromise {
	return s.asyncLoadBlockRecord(channelID)
}

func (s *store) Block(channelID string) error {
	rec, err := s.asyncLoadBlockRecord(channelID).block()
	if err != nil {
		return err
	}

	if rec == nil {
		rec = new(blockRecord)
		rec.ChannelID = channelID
	}

	// idempotence
	if rec.Activated != nil {
		return nil
	}

	now := time.Now()
	rec.Activated = &now
	rec.DeletingSince = &now
	return s.saveBlockRecord(rec)
}

func (s *store) Unblock(channelID string) error {
	rec, err := s.asyncLoadBlockRecord(channelID).block()
	if err != nil {
		return err
	}

	// idempotence
	if rec == nil || rec.Activated == nil {
		return nil
	}

	rec.Activated = nil
	rec.DeletingSince = nil
	return s.saveBlockRecord(rec)
}

// This will remove the record from the `deletion_index` so that it is no longer
// visible to any catch-up job.
func (s *store) OnDeletionFinished(channelID string) error {
	rec, err := s.asyncLoadBlockRecord(channelID).block()
	if err != nil {
		return err
	}

	// idempotence
	if rec == nil || rec.DeletingSince == nil {
		return nil
	}

	rec.DeletingSince = nil
	return s.saveBlockRecord(rec)
}

// This method is performing a scan over a sparse index -- only records that contain
// the key DeletingSince will be present in the index, so we can brute force the
// entire space. Paging has been added since this is a potentially unbounded bulk
// call.
func (s *store) DeletionInProgress() ([]string, error) {
	input := new(dynamodb.ScanInput)
	input.SetConsistentRead(true)
	input.SetTableName(blockTable(s.prefix))
	input.SetIndexName("deletion_index")
	input.SetAttributesToGet([]*string{aws.String(fieldChannelID)})

	channels := []string{}
	err := s.db.ScanPages(input, func(page *dynamodb.ScanOutput, lastPage bool) bool {
		for _, v := range page.Items {
			if v[fieldChannelID] != nil && v[fieldChannelID].S != nil {
				channels = append(channels, *v[fieldChannelID].S)
			}
		}
		return !lastPage
	})

	if err != nil {
		return nil, err
	}
	return channels, nil
}

func (s *store) asyncLoadBlockRecord(channelID string) *blockPromise {
	inner := s.reader.Add(blockTable(s.prefix), false, map[string]*dynamodb.AttributeValue{
		fieldChannelID: {S: aws.String(channelID)},
		fieldRangeKey:  {S: aws.String(valueRangeKey)},
	})
	return newBlockPromise(inner, s.unmarshal)
}

func (s *store) saveBlockRecord(rec *blockRecord) error {
	entry, uuid, err := MarshalForSave(rec, s.uuid, s.marshal)
	if err != nil {
		return err
	}
	params := &dynamodb.PutItemInput{
		TableName: aws.String(blockTable(s.prefix)),
		Item:      entry,
	}
	att := make(map[string]*dynamodb.AttributeValue)

	if rec.ConcurrencyUUID == "" {
		params.SetConditionExpression(condExpCreateBlock)
	} else {
		att[conditionExpectedUUID] = new(dynamodb.AttributeValue).SetS(rec.ConcurrencyUUID)
		params.SetConditionExpression(condExpUpdate)
		params.SetExpressionAttributeValues(att)
	}

	_, err = s.db.PutItem(params)
	if err == nil {
		rec.ConcurrencyUUID = uuid
	}
	return err
}

func blockTable(prefix string) string { return prefix + "_blocked" }
