package dynamo

import (
	"testing"

	"github.com/stretchr/testify/assert"

	"code.justin.tv/extensions/configuration/services/main/data/model"
	"code.justin.tv/extensions/configuration/services/main/data/model/bad"
	"code.justin.tv/gds/gds/golibs/dynamodb/lazy"
)

func TestChannelPromise(t *testing.T) {
	t.Run("should handle unset correctly", func(t *testing.T) {
		d := new(dummyPromise)
		d.block = true
		p := newChannelPromise(d, d, lazy.UnmarshalMap)
		assert.True(t, p.WouldBlock())
	})

	t.Run("should handle set correctly", func(t *testing.T) {
		src := model.NewChannel("env", "extID", "chID")
		d := new(dummyPromise)
		d.data, d.err = lazy.MarshalMap(src)
		p := newChannelPromise(d, d, lazy.UnmarshalMap)
		assert.False(t, p.WouldBlock())

		i, e := p.Get()
		assert.Equal(t, src, i)
		assert.NoError(t, e)

		out, e := p.Channel()
		assert.Equal(t, src, out)
		assert.NoError(t, e)
	})

	t.Run("should handle empty correctly", func(t *testing.T) {
		d := new(dummyPromise)
		d.data, d.err = lazy.MarshalMap(nil)
		p := newChannelPromise(d, d, lazy.UnmarshalMap)
		assert.False(t, p.WouldBlock())
		i, e := p.Get()
		assert.Nil(t, i)
		assert.NoError(t, e)

		out, e := p.Channel()
		assert.Nil(t, out)
		assert.NoError(t, e)
	})

	t.Run("should handle blocked correctly", func(t *testing.T) {
		src := model.NewChannel("env", "extID", "chID")
		d := new(dummyPromise)
		d.data, d.err = lazy.MarshalMap(src)
		d.active = true
		p := newChannelPromise(d, d, lazy.UnmarshalMap)
		assert.False(t, p.WouldBlock())

		i, e := p.Get()
		assert.Empty(t, i)
		assert.NoError(t, e)

		out, e := p.Channel()
		assert.Empty(t, out)
		assert.NoError(t, e)
	})

	t.Run("should handle errored correctly", func(t *testing.T) {
		d := &dummyPromise{err: bad.ErrExpected}
		p := newChannelPromise(d, d, lazy.UnmarshalMap)
		assert.False(t, p.WouldBlock())
		i, e := p.Get()
		assert.Nil(t, i)
		assert.Equal(t, bad.ErrExpected, e)

		out, e := p.Channel()
		assert.Nil(t, out)
		assert.Equal(t, bad.ErrExpected, e)
	})

	t.Run("should handle errored block promise correctly", func(t *testing.T) {
		d := &dummyPromise{err: bad.ErrExpected}
		d2 := &dummyPromise{}
		p := newChannelPromise(d, d2, lazy.UnmarshalMap)
		assert.False(t, p.WouldBlock())
		i, e := p.Get()
		assert.Nil(t, i)
		assert.Equal(t, bad.ErrExpected, e)

		out, e := p.Channel()
		assert.Nil(t, out)
		assert.Equal(t, bad.ErrExpected, e)
	})
}
