package dynamo

import (
	"time"

	"code.justin.tv/extensions/configuration/services/main/data/model"
	"code.justin.tv/extensions/configuration/services/main/data/model/shared"
	"code.justin.tv/extensions/configuration/services/main/protocol"
	"code.justin.tv/gds/gds/golibs/awsutil"
	"code.justin.tv/gds/gds/golibs/dynamodb/aggregator"
	"code.justin.tv/gds/gds/golibs/dynamodb/lazy"
	"code.justin.tv/gds/gds/golibs/uuid"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
)

const (
	conditionExpectedUUID   = ":uuid"
	condExpCreateBlock      = "attribute_not_exists(" + fieldChannelID + ")"
	condExpCreateChannel    = "attribute_not_exists(" + fieldChannelID + ")"
	condExpCreateCommon     = "attribute_not_exists(" + fieldExtensionID + ")"
	condExpUpdate           = fieldUUID + " = " + conditionExpectedUUID
	condExpMarkPublished    = fieldUUID + " = " + conditionExpectedUUID + " AND attribute_exists(" + fieldUnpublished + ")"
	expressionMarkPublished = "REMOVE " + fieldUnpublished + ", " + fieldMessages
	fieldChannelID          = "ChannelID"
	fieldEnvironment        = "Environment"
	fieldEnvAndExtID        = "EnvAndExtID"
	fieldExtensionID        = "ExtensionID"
	fieldMessages           = "Messages"
	fieldRangeKey           = "X" // required structurally for sparse index to be available
	fieldUUID               = "ConcurrencyUUID"
	fieldUnpublished        = "UnpublishedTime"
	valueRangeKey           = "X"
)

type store struct {
	uuid         uuid.Source
	prefix       string
	db           dynamodbiface.DynamoDBAPI
	reader       *aggregator.Read
	resetEnabled int32
	tracker      model.BlockTracker
	marshal      MarshalFunc
	unmarshal    UnmarshalFunc
}

func New(src uuid.Source, db dynamodbiface.DynamoDBAPI, tracker model.BlockTracker, prefix string, batchTime time.Duration) model.StoreWithTracker {
	out := &store{
		uuid:      src,
		prefix:    prefix,
		db:        db,
		reader:    aggregator.NewRead(db, batchTime),
		marshal:   lazy.MarshalMap,
		unmarshal: lazy.UnmarshalMap,
		tracker:   tracker,
	}
	out.reader.SetErrNotRead(nil) // lack of record is an acceptible outcome for this service
	if tracker == nil {
		out.tracker = out
	}
	return out
}

func (s *store) AsyncLoadCommon(env, extID string) model.CommonPromise {
	inner := s.reader.Add(commonTable(s.prefix), false, map[string]*dynamodb.AttributeValue{
		fieldExtensionID: {S: aws.String(extID)},
		fieldEnvironment: {S: aws.String(env)},
	})
	return shared.CommonPromise(newCommonPromise(inner, s.unmarshal))
}

func (s *store) LoadCommon(env, extID string) (*model.Common, error) {
	return s.AsyncLoadCommon(env, extID).Common()
}

func (s *store) AsyncLoadChannel(env, extID, chID string) model.ChannelPromise {
	inner := s.reader.Add(channelsTable(s.prefix), false, map[string]*dynamodb.AttributeValue{
		fieldChannelID:   {S: aws.String(chID)},
		fieldEnvAndExtID: {S: envAndExtID(env, extID)},
	})
	return shared.ChannelPromise(newChannelPromise(inner, s.tracker.IsBlocked(chID), s.unmarshal))
}

func (s *store) LoadChannel(env, extID, chID string) (*model.Channel, error) {
	return s.AsyncLoadChannel(env, extID, chID).Channel()
}

func (s *store) SaveCommon(c *model.Common) error {
	entry, uuid, err := MarshalForSave(c, s.uuid, s.marshal)
	if err != nil {
		return err
	}
	params := &dynamodb.PutItemInput{
		TableName: aws.String(commonTable(s.prefix)),
		Item:      entry,
	}
	att := make(map[string]*dynamodb.AttributeValue)

	if c.ConcurrencyUUID == "" {
		params.SetConditionExpression(condExpCreateCommon)
	} else {
		att[conditionExpectedUUID] = new(dynamodb.AttributeValue).SetS(c.ConcurrencyUUID)
		params.SetConditionExpression(condExpUpdate)
		params.SetExpressionAttributeValues(att)
	}

	_, err = s.db.PutItem(params)
	if err == nil {
		c.ConcurrencyUUID = uuid
	}
	return wrap(err)
}

func (s *store) SaveChannel(c *model.Channel) error {
	entry, uuid, err := MarshalForSave(c, s.uuid, s.marshal)
	if err != nil {
		return err
	}
	params := &dynamodb.PutItemInput{
		TableName: aws.String(channelsTable(s.prefix)),
		Item:      entry,
	}
	att := make(map[string]*dynamodb.AttributeValue)

	if c.ConcurrencyUUID == "" {
		params.SetConditionExpression(condExpCreateChannel)
	} else {
		att[conditionExpectedUUID] = new(dynamodb.AttributeValue).SetS(c.ConcurrencyUUID)
		params.SetConditionExpression(condExpUpdate)
		params.SetExpressionAttributeValues(att)
	}

	active, err := s.tracker.IsBlocked(c.ChannelID).Active()
	if err != nil {
		return err
	}
	if active {
		return protocol.ErrForbiddenByBroadcaster
	}

	_, err = s.db.PutItem(params)
	if err == nil {
		c.ConcurrencyUUID = uuid
	}
	return wrap(err)
}

func (s *store) DeleteChannel(channelID string) error {
	return s.tracker.Block(channelID)
}

func (s *store) MarkCommonPublished(c *model.Common) error {
	key := map[string]*dynamodb.AttributeValue{
		fieldExtensionID: {S: aws.String(c.ExtensionID)},
		fieldEnvironment: {S: aws.String(c.Environment)},
	}
	return s.markPublished(commonTable(s.prefix), key, c.ConcurrencyUUID)
}

func (s *store) MarkChannelPublished(c *model.Channel) error {
	key := map[string]*dynamodb.AttributeValue{
		fieldChannelID:   {S: aws.String(c.ChannelID)},
		fieldEnvAndExtID: {S: envAndExtID(c.Environment, c.ExtensionID)},
	}
	return s.markPublished(channelsTable(s.prefix), key, c.ConcurrencyUUID)
}

func (s *store) markPublished(table string, key map[string]*dynamodb.AttributeValue, uuid string) error {
	params := new(dynamodb.UpdateItemInput)
	params.SetTableName(table)
	params.SetKey(key)
	params.SetUpdateExpression(expressionMarkPublished)
	params.SetConditionExpression(condExpMarkPublished)
	params.SetExpressionAttributeValues(map[string]*dynamodb.AttributeValue{
		conditionExpectedUUID: new(dynamodb.AttributeValue).SetS(uuid),
	})

	_, err := s.db.UpdateItem(params)
	return wrap(err)
}

func channelsTable(prefix string) string    { return prefix + "_channels" }
func commonTable(prefix string) string      { return prefix + "_common" }
func envAndExtID(env, extID string) *string { return aws.String(env + ":" + extID) }
func wrap(err error) error {
	if awsutil.GetCode(err) == dynamodb.ErrCodeConditionalCheckFailedException {
		return protocol.ErrConcurrency
	}
	return err
}
