package memory

import (
	"testing"

	"code.justin.tv/extensions/configuration/services/main/data/model"
	"code.justin.tv/extensions/configuration/services/main/data/model/bad"
	"code.justin.tv/extensions/configuration/services/main/protocol"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func testValue(t *testing.T, ext, ch string, seg protocol.SegmentType, p model.RecordPromise) {
	require.False(t, p.WouldBlock())
	addr := protocol.Address{
		ChannelID:   ch,
		ExtensionID: ext,
		SegmentType: seg,
	}

	a, r, e := p.Record()
	assert.Equal(t, &addr, a)
	assert.Nil(t, r)
	assert.NoError(t, e)

	m, e := p.Get()
	assert.Equal(t, protocol.RecordMap{addr: r}, m)
	assert.Nil(t, e)
}

func TestChannelPromise(t *testing.T) {
	t.Run("should handle unset correctly", func(t *testing.T) {
		p := newChannelPromise()
		assert.True(t, p.WouldBlock())
	})

	t.Run("should handle set correctly", func(t *testing.T) {
		p := newChannelPromise()
		in := model.NewChannel("env", "extID", "chID")
		p.inner.Set(in, nil)
		assert.False(t, p.WouldBlock())
		i, e := p.Get()
		assert.Equal(t, in, i)
		assert.NoError(t, e)

		out, e := p.Channel()
		assert.Equal(t, in, out)
		assert.NoError(t, e)
	})

	t.Run("should handle mis-set correctly", func(t *testing.T) {
		p := newChannelPromise()
		p.inner.Set("wrong", nil)
		assert.False(t, p.WouldBlock())
		i, e := p.Get()
		assert.Equal(t, "wrong", i)
		assert.NoError(t, e)

		out, e := p.Channel()
		assert.Nil(t, out)
		assert.Equal(t, protocol.ErrUnavailable, e)
	})

	t.Run("should handle empty correctly", func(t *testing.T) {
		p := newChannelPromise()
		p.inner.Set(nil, nil)
		assert.False(t, p.WouldBlock())
		i, e := p.Get()
		assert.Nil(t, i)
		assert.NoError(t, e)

		out, e := p.Channel()
		assert.Nil(t, out)
		assert.NoError(t, e)
	})

	t.Run("should handle errored correctly", func(t *testing.T) {
		p := newChannelPromise()
		p.inner.Set(nil, bad.ErrExpected)
		assert.False(t, p.WouldBlock())
		i, e := p.Get()
		assert.Nil(t, i)
		assert.Equal(t, bad.ErrExpected, e)

		out, e := p.Channel()
		assert.Nil(t, out)
		assert.Equal(t, bad.ErrExpected, e)
	})
}
