package protocol

import (
	"net/http"
	"testing"

	"code.justin.tv/gds/gds/golibs/errors"
	"github.com/stretchr/testify/assert"
)

func TestErrUnknownSegmentType(t *testing.T) {
	err := ErrUnknownSegmentType(SegmentType("???"))
	assert.Error(t, err)
	assert.Equal(t, http.StatusUnprocessableEntity, errors.GetHTTPStatusOrDefault(err, 0))
	assert.Equal(t, ErrUnknownSegmentTypeCode, errors.GetErrorCode(err))
	assert.Equal(t, errors.Details{"type": SegmentType("???")}, errors.GetDetails(err))
	assert.Equal(t, "Unknown segment type \"???\"", err.Error())
}

func TestErrIllegalSegmentChannel(t *testing.T) {
	err := ErrIllegalSegmentChannel(GlobalType, "chID")
	assert.Error(t, err)
	assert.Equal(t, http.StatusUnprocessableEntity, errors.GetHTTPStatusOrDefault(err, 0))
	assert.Equal(t, ErrIllegalSegmentChannelCode, errors.GetErrorCode(err))
	assert.Equal(t, errors.Details{"type": GlobalType, "channel": "chID"}, errors.GetDetails(err))
	assert.Equal(t, "Segment type \"global\" is incompatible with channel \"chID\"", err.Error())
}

func TestErrIllegalSegmentLength(t *testing.T) {
	addr := Address{ExtensionID: "extID", SegmentType: "global"}
	err := ErrIllegalSegmentLength(addr, 1<<16)
	assert.Error(t, err)
	assert.Equal(t, http.StatusUnprocessableEntity, errors.GetHTTPStatusOrDefault(err, 0))
	assert.Equal(t, ErrIllegalSegmentLengthCode, errors.GetErrorCode(err))
	assert.Equal(t, errors.Details{
		"channel_id":   addr.ChannelID,
		"extension_id": addr.ExtensionID,
		"length":       1 << 16,
		"max_length":   MaxSegmentLength,
		"type":         addr.SegmentType,
	}, errors.GetDetails(err))
	assert.Equal(t, "Segment \"global\" length 65536 > maximum of 5120 for extension \"extID\"", err.Error())

	addr = Address{ExtensionID: "extID", SegmentType: "broadcaster", ChannelID: "chID"}
	err = ErrIllegalSegmentLength(addr, 1<<16)
	assert.Equal(t, errors.Details{
		"channel_id":   addr.ChannelID,
		"extension_id": addr.ExtensionID,
		"length":       1 << 16,
		"max_length":   MaxSegmentLength,
		"type":         addr.SegmentType,
	}, errors.GetDetails(err))
	assert.Equal(t, "Segment \"broadcaster\" length 65536 > maximum of 5120 for extension \"extID\", channel \"chID\"", err.Error())
}
