package messages

import (
	"encoding/json"
	"time"

	"code.justin.tv/extensions/configuration/services/main/protocol"
	"code.justin.tv/gds/gds/golibs/event"
)

const (
	ConfigTopic = event.Topic("Config")

	OnSet = event.Change("Set")
)

type ConfigMessage struct {
	Format    int              `json:"format_version"`
	Sequence  int              `json:"sequence"`
	UserID    string           `json:"user_id"`
	Time      time.Time        `json:"time"`
	Operation string           `json:"operation"`
	Address   protocol.Address `json:"address"`
	Record    *protocol.Record `json:"record"`
}

var _ event.Message = (*ConfigMessage)(nil)

func NewConfigMessage(seq int, change event.Change) *ConfigMessage {
	return &ConfigMessage{Format: 1, Sequence: seq, Operation: string(change), Time: time.Now().Truncate(time.Second)}
}

// LoadConfigMessage deserializes an existing config message
func LoadConfigMessage(change event.Change, format int, body []byte) (event.Message, error) {
	ch := &ConfigMessage{}
	if err := json.Unmarshal(body, ch); err != nil {
		return nil, err
	}
	ch.Operation = string(change)
	return ch, nil
}

func (c *ConfigMessage) FormatVersion() int   { return c.Format }
func (c *ConfigMessage) Topic() event.Topic   { return ConfigTopic }
func (c *ConfigMessage) Change() event.Change { return event.Change(c.Operation) }
func (c *ConfigMessage) Body() interface{}    { return c }
