package protocol

import (
	"encoding/json"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestRecordMap_UnmarshalJSON(t *testing.T) {
	t.Run("should unmarshal empty correctly", func(t *testing.T) {
		src := RecordMap{}
		assert.NoError(t, src.UnmarshalJSON([]byte("{}")))
	})

	t.Run("should unmarshal valid correctly", func(t *testing.T) {
		content := "blah"
		src := RecordMap{
			Address{
				ExtensionID: "extID",
				SegmentType: GlobalType,
				ChannelID:   "chID",
			}: NewRecord("1", &content),
		}
		data, err := src.MarshalJSON()
		require.NoError(t, err)
		assert.NoError(t, src.UnmarshalJSON(data))
	})

	t.Run("should report invalid unmarshal errors correctly", func(t *testing.T) {
		src := RecordMap{}
		assert.IsType(t, &json.SyntaxError{}, src.UnmarshalJSON([]byte("{")))
	})
}

func TestRecordMapToChannelReturn(t *testing.T) {
	content := "blah"
	t.Run("should combine records for an extension", func(t *testing.T) {
		r0 := RecordMap(map[Address]*Record{Address{
			ExtensionID: "testid",
			SegmentType: BroadcasterType,
			ChannelID:   "testChannel",
		}: NewRecord("blah", &content), Address{
			ExtensionID: "testid",
			SegmentType: DeveloperType,
			ChannelID:   "testChannel",
		}: NewRecord("blah", &content), Address{
			ExtensionID: "testid",
			SegmentType: GlobalType,
		}: NewRecord("blah", &content)})
		channel, err := RecordMapToChannelReturn(r0)
		assert.Nil(t, err)
		assert.Len(t, channel, 1)
		assert.Len(t, channel["testid"], 3)
		assert.Contains(t, channel["testid"], BroadcasterType)
		assert.Contains(t, channel["testid"], DeveloperType)
		assert.Contains(t, channel["testid"], GlobalType)
	})

	t.Run("should error with multiple channelIds", func(t *testing.T) {
		r0 := RecordMap(map[Address]*Record{Address{
			ExtensionID: "testid",
			SegmentType: BroadcasterType,
			ChannelID:   "testChannel",
		}: NewRecord("blah", &content), Address{
			ExtensionID: "testid",
			SegmentType: GlobalType,
		}: NewRecord("blah", &content), Address{
			ExtensionID: "testid",
			SegmentType: DeveloperType,
			ChannelID:   "testChannel2",
		}: NewRecord("blah", &content)})
		channel, err := RecordMapToChannelReturn(r0)
		assert.Equal(t, err, ErrChannelIDMismatch)
		assert.Nil(t, channel)
	})

	t.Run("should handle multiple extensions", func(t *testing.T) {
		r0 := RecordMap(map[Address]*Record{Address{
			ExtensionID: "testid",
			SegmentType: BroadcasterType,
			ChannelID:   "testChannel",
		}: NewRecord("blah", &content), Address{
			ExtensionID: "testid2",
			SegmentType: GlobalType,
		}: NewRecord("blah", &content), Address{
			ExtensionID: "testid2",
			SegmentType: DeveloperType,
			ChannelID:   "testChannel",
		}: NewRecord("blah", &content)})
		channel, err := RecordMapToChannelReturn(r0)
		assert.Nil(t, err)
		assert.Len(t, channel, 2)
		assert.Len(t, channel["testid"], 1)
		assert.Contains(t, channel["testid"], BroadcasterType)
		assert.Len(t, channel["testid2"], 2)
		assert.Contains(t, channel["testid2"], DeveloperType)
		assert.Contains(t, channel["testid2"], GlobalType)
	})

}
