package protocol

import (
	"encoding/json"
)

type SegmentType string

const (
	DeveloperType   = SegmentType("developer")
	BroadcasterType = SegmentType("broadcaster")
	GlobalType      = SegmentType("global")

	MaxSegmentLength = 5 * 1024
)

// Segment defines a location inside an extension's hosted configuration space
type Segment = *segmentData

// Type defines the scope and access rights for a segment
func (s *segmentData) Type() SegmentType {
	if s == nil {
		return GlobalType
	}
	return s.segmentType
}

// ChannelID defines which broadcaster is targeted by a segment; the zero value
// indicates a cross-channel segment.
func (s *segmentData) ChannelID() string {
	if s == nil {
		return ""
	}
	return s.channelID
}

func (s *segmentData) Address(extensionID string) Address {
	return Address{extensionID, s.Type(), s.ChannelID()}
}

// The Global Segment is the zero value. It is cross-channel but extension specific
func Global() Segment { return &segmentData{segmentType: GlobalType} }

// Broadcaster segments are channel specific.  They can be written by the developer or broadcaster
func Broadcaster(channelID string) (Segment, error) {
	if channelID == "" {
		return nil, ErrIllegalSegmentChannel(BroadcasterType, channelID)
	}
	return &segmentData{BroadcasterType, channelID}, nil
}

// Developer segments are channel specific.  They can only be written by the developer
func Developer(channelID string) (Segment, error) {
	if channelID == "" {
		return nil, ErrIllegalSegmentChannel(DeveloperType, channelID)
	}
	return &segmentData{DeveloperType, channelID}, nil
}

// segmentData describes a configuration address inside the scope of an extension
type segmentData struct {
	segmentType SegmentType
	channelID   string
}

// SegmentsForChannel returns the list of segments necessary to load each extension on the
// indicated channel
func SegmentsForChannel(channelID string, includeCommon bool) ([]*segmentData, error) {
	var broad Segment
	dev, err := Developer(channelID)
	if err == nil {
		// this can't independently fail if Developer succeeds today;
		// leaving the error check explicit allows that to change
		// without introducing a bug.
		broad, err = Broadcaster(channelID)
	}
	if err != nil {
		return nil, err
	}
	if includeCommon {
		return []*segmentData{dev, broad, Global()}, nil
	}
	return []*segmentData{dev, broad}, nil
}

// TODO : equality, sort, customized serialization so that iternals remain package private

func (s *segmentData) String() string {
	return string(s.Type()) + ":" + s.ChannelID()
}

func (s *segmentData) MarshalJSON() ([]byte, error) {
	return json.Marshal(&struct {
		SegmentType SegmentType `json:"segment_type"`
		ChannelID   string      `json:"channel_id,omitempty"`
	}{
		ChannelID:   s.ChannelID(),
		SegmentType: s.Type(),
	})
}

func (s *segmentData) UnmarshalJSON(data []byte) error {
	aux := &struct {
		SegmentType SegmentType `json:"segment_type"`
		ChannelID   string      `json:"channel_id"`
	}{}
	if err := json.Unmarshal(data, &aux); err != nil {
		return err
	}
	s.channelID = aux.ChannelID
	s.segmentType = aux.SegmentType
	return nil
}
