package configuration

import (
	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/gds/gds/golibs/config"
	htmid "code.justin.tv/gds/gds/golibs/http"
	hmid "code.justin.tv/gds/gds/golibs/hystrix"
	"code.justin.tv/gds/gds/golibs/metrics"
	"code.justin.tv/infosec/cors"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"
	goji "goji.io"
	"goji.io/pat"

	"code.justin.tv/extensions/configuration/services/main/api"
	"code.justin.tv/extensions/configuration/services/main/auth"
	"code.justin.tv/extensions/configuration/services/main/data"
)

const (
	ConfigStatsFrequent = "stats.frequency.frequent"
	ConfigStatsCommon   = "stats.frequency.common"
	ConfigStatsRare     = "stats.frequency.rare"
)

var (
	corsPolicy = cors.Policy{
		AllowedOrigins: cors.Origins("*"),
		AllowMethods:   cors.Methods("GET", "POST", "PUT", "DELETE"),
		AllowHeaders:   cors.Headers("Accept", "Authorization", "Client-ID", "Twitch-Api-Token", "X-Forwarded-Proto", "X-Requested-With", "X-CSRF-Token", "Content-Type"),
	}
)

// BuildServer combines all parts that make up smart to create functioning handlers
func BuildServer(a *api.API, stats statsd.Statter, hystrixTimeout int, cfg config.Config, handler auth.Handler, mgr data.ConfigurationManager) *goji.Mux {
	instr := metrics.NewStattingInstrumentor(stats, map[metrics.Frequency]float32{
		metrics.Common:   float32(cfg.GetFloat(ConfigStatsCommon, 0.1)),
		metrics.Frequent: float32(cfg.GetFloat(ConfigStatsFrequent, 0.01)),
		metrics.Rare:     float32(cfg.GetFloat(ConfigStatsRare, 1)),
	})

	server := twitchserver.NewServer()
	server.Use(corsPolicy.MustMiddleware)
	server.Use(cors.BlockOnOptions)
	server.Use(metrics.Instrument(instr, "unknown", metrics.Common))
	server.Use(data.StoreErrorConverter())
	server.Use(hmid.CreateCircuit("unknown"))
	server.Use(config.StoreConfig(cfg))
	server.Use(auth.StoreHandler(handler))
	server.Use(data.StoreManager(mgr))

	hystrix.DefaultTimeout = hystrixTimeout
	hystrix.DefaultMaxConcurrent = 2000
	hystrix.DefaultSleepWindow = 1000
	hystrix.DefaultErrorPercentThreshold = 60

	AppendFunctions(a, server)

	return server
}

func AppendFunctions(a *api.API, mux *goji.Mux) {
	mux.Handle(pat.Get("/status/config"), htmid.NewContextHandler("config", metrics.Rare, config.Display))
	mux.Handle(pat.Get("/channel/:cid"), htmid.NewAPIHandler("get_channel_configuration", metrics.Frequent, a.GetChannelConfiguration))
	mux.Handle(pat.Put("/channel/:cid"), htmid.NewAPIHandler("set_channel_configuration", metrics.Common, a.SetChannelConfiguration))
	mux.Handle(pat.Delete("/channel/:cid"), htmid.NewAPIHandler("delete_channel_configuration", metrics.Rare, a.DeleteConfiguration))
	mux.Handle(pat.Post("/addresses"), htmid.NewAPIHandler("get_configuration", metrics.Common, a.GetConfiguration))
	mux.Handle(pat.Put("/addresses"), htmid.NewAPIHandler("set_configuration", metrics.Common, a.SetConfiguration))

	// V2  Routes
	mux.Handle(pat.Delete("/channel/:cid/v2"), htmid.NewAPIHandler("delete_channel_configuration", metrics.Rare, a.DeleteConfigurationV2))
	mux.Handle(pat.Put("/channel/:cid/v2"), htmid.NewAPIHandler("set_channel_configuration", metrics.Common, a.SetChannelConfigurationV2))
	mux.Handle(pat.Put("/addresses/v2"), htmid.NewAPIHandler("set_configuration", metrics.Common, a.SetConfigurationV2))
}
