package protocol

import "encoding/json"

// Record is an immutable container that associates a data blob with a version tag
type Record struct {
	content *string
	version string
}

var _ json.Marshaler = (*Record)(nil)
var _ json.Unmarshaler = (*Record)(nil)

// NewRecord creates a new value
func NewRecord(version string, content *string) *Record {
	return &Record{content, version}
}

// Content is the developer specified data blob held by this record
func (r *Record) Content() *string { return r.content }

// Length is the Length of the developer specified data blob held by this record
func (r *Record) Length() int {
	if r.content == nil {
		return 0
	}
	return len(*r.content)
}

// Version is the developer specified version string held by this record
func (r *Record) Version() string { return r.version }

// MarshalJSON implements json.Marshaler
func (r *Record) MarshalJSON() ([]byte, error) {
	values := map[string]interface{}{}
	if r.Length() > 0 {
		values["content"] = r.content
	}
	if len(r.version) > 0 {
		values["version"] = r.version
	}
	return json.Marshal(values)
}

// UnmarshalJSON implements json.Unmarshaler
func (r *Record) UnmarshalJSON(data []byte) error {
	values := struct {
		Content *string `json:"content,omitempty"`
		Version string  `json:"version,omitempty"`
	}{}
	if err := json.Unmarshal(data, &values); err != nil {
		return err
	}
	r.content = values.Content
	r.version = values.Version
	return nil
}

// RecordEqual compares two records
func RecordEqual(r *Record, o *Record) bool {
	if r == nil || o == nil {
		return r == o
	}
	if r.version != o.version {
		return false
	}
	if r.content == o.content {
		return true
	}
	if r.content == nil || o.content == nil {
		return false
	}
	if *r.content != *o.content {
		return false
	}
	return true
}
