package awsutil

import (
	"net"
	"net/http"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
)

// GetCode extracts an Amazon error code from an error if present or returns
// empty string otherwise.
func GetCode(err error) string {
	if err != nil {
		if awsErr, ok := err.(awserr.Error); ok {
			return awsErr.Code()
		}
	}
	return ""
}

// CreateServiceConfig generates a config that uses standard settings for
// extension services. The HTTP client settings here are based on tuning that
// was done for OWL connections to dynamoDB.
func CreateServiceConfig(region string) *aws.Config {
	return aws.NewConfig().
		WithRegion(region).
		WithCredentialsChainVerboseErrors(true).
		WithHTTPClient(&http.Client{
			Timeout: 1 * time.Second,
			Transport: &http.Transport{
				Proxy: http.ProxyFromEnvironment,
				DialContext: (&net.Dialer{
					Timeout:   1 * time.Second,
					KeepAlive: 30 * time.Second,
				}).DialContext,
				TLSHandshakeTimeout: 10 * time.Second,
				MaxIdleConnsPerHost: 500,
			},
		})
}
