package config

import (
	"encoding/json"

	"code.justin.tv/gds/gds/golibs/config/sources"

	"github.com/aws/aws-sdk-go/service/ec2/ec2iface"
	"github.com/aws/aws-sdk-go/service/s3/s3iface"
)

// Source is a convenience wrapper around config/sources.Source
type Source sources.Source

// RefreshableSource is a convenience wrapper around config/sources.RefreshableSource
type RefreshableSource sources.RefreshableSource

// NewConfig returns a Config given a list of Sources
func NewConfig(src ...sources.Source) Config {
	return &config{src}
}

// NewRefreshController returns a new refresh controller
func NewRefreshController() RefreshController {
	return &refreshController{}
}

// FromJSON returns a static Source from a JSON string; nested structures are
// converted into dotted sections.
func FromJSON(name string, data []byte) (Source, error) {
	var values map[string]interface{}
	if err := json.Unmarshal(data, &values); err != nil {
		return nil, err
	}
	return sources.NewStaticSource(name, values)
}

// FromRawValues returns a static Source from an arbitrary map of string/value
// pairs. If any keys point to unsupported value types, an error will be
// returned.  If a nested map is detected, it will form a dotted namespace
// inside of the top level.
func FromRawValues(name string, values map[string]interface{}) (Source, error) {
	return sources.NewStaticSource(name, values)
}

// FromEC2 returns a RefreshableSource that is populated from the EC2 tags on
// the current instance if applicable. The config argument can be used to
// override the prefix used to populate the result.
func FromEC2(config sources.Settings, instanceID string, service ec2iface.EC2API) (RefreshableSource, error) {
	return sources.NewRefreshableSource(config, sources.NewEC2RefreshLogic(instanceID, service))
}

// FromS3 returns a RefreshableSource that loads JSON from an S3 bucket using
// settings that are determined by an "upstream" Config
func FromS3(config sources.Settings, s3 s3iface.S3API) (RefreshableSource, error) {
	return sources.NewRefreshableSource(config, sources.NewS3RefreshLogic(s3))
}

// FromEnv returns a RefreshableSource that loads individual string values from
// the operating system environment; the names of the variables to read must be
// explicitly set.
func FromEnv(config sources.Settings, names []string) (RefreshableSource, error) {
	return sources.NewRefreshableSource(config, sources.NewEnvironmentRefreshLogic(names))
}

// FromFilesystem returns a RefreshableSource that loads JSON from a local file
// using the specified path.
func FromFilesystem(config sources.Settings, path string) (RefreshableSource, error) {
	return sources.NewRefreshableSource(config, sources.NewFilesystemRefreshLogic(path))
}
