package builders

import (
	"fmt"
	"log"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws/client"
	"github.com/aws/aws-sdk-go/aws/ec2metadata"
	"github.com/aws/aws-sdk-go/service/ec2"
	"github.com/aws/aws-sdk-go/service/s3"

	"code.justin.tv/gds/gds/golibs/awsutil"
	"code.justin.tv/gds/gds/golibs/config"
	"code.justin.tv/gds/gds/golibs/config/sources"
)

type SourceType string

const (
	S3     SourceType = "s3"
	EC2               = "ec2"
	File              = "file"
	Custom            = "custom"

	localDebugPath string = "config/local_debug.config.json"
)

type Source struct {
	// for standard config types only the following block is required
	Type    SourceType
	Refresh time.Duration

	// for custom types the above are needed as well as the following
	Callback func(config.Config, client.ConfigProvider) sources.RefreshableSource
	Name     string

	// Storing this allows for Scheduling
	source sources.RefreshableSource
}

type StandardDefaults struct {
	ComponentName      string
	Env                string
	Name               string
	AdditionalDefaults map[string]interface{}
	Sources            []*Source
}

// CreateConfigAndSchedule should be used when only the standard Config is required
func (d *StandardDefaults) CreateConfigAndSchedule(cp client.ConfigProvider) (config.Config, config.RefreshController) {
	cfg := d.createConfig(cp)
	ctrl := d.scheduleUpdates(cfg)
	return cfg, ctrl
}

// CreateConfig should be used when additional config with Scheduling is needed beyond the standard setup
// ScheduleUpdates must be called after cfg is updated to schedule the updates
func (d *StandardDefaults) createConfig(cp client.ConfigProvider) config.Config {
	if cp == nil {
		panic("AWS must be configured")
	}
	if d.ComponentName == "" || d.Env == "" || d.Name == "" {
		panic("ComponentName, Env, and Name must be set")
	}
	defaultsMap := map[string]interface{}{}
	if d.AdditionalDefaults != nil {
		for k, v := range d.AdditionalDefaults {
			defaultsMap[k] = v
		}
	}
	defaultsMap["app.component_name"] = d.ComponentName
	defaultsMap["app.env"] = d.Env
	defaultsMap["app.name"] = d.Name
	defaults, err := config.FromRawValues("defaults", defaultsMap)

	if err != nil {
		panic(err)
	}
	cfg := config.NewConfig(defaults)
	for _, s := range d.Sources {
		switch s.Type {
		case Custom:
			cus := s.Callback(cfg, cp)
			if cus != nil {
				cfg = cfg.WithOverrides(cus)
				s.source = cus
			}
		case EC2:
			ec2 := createEC2Source(cfg, cp)
			if ec2 != nil {
				cfg = cfg.WithOverrides(ec2)
				s.source = ec2
			}
		case File:
			file := createFileSource(cfg, cp)
			if file != nil {
				cfg = cfg.WithOverrides(file)
				s.source = file
			}
		case S3:
			s3 := createS3Source(cfg, cp)
			if s3 != nil {
				cfg = cfg.WithOverrides(s3)
				s.source = s3
			}
		}
	}

	return cfg
}

func createEC2Source(cfg sources.Settings, cp client.ConfigProvider) sources.RefreshableSource {
	meta := ec2metadata.New(cp)
	if !meta.Available() {
		return nil
	}
	doc, err := meta.GetInstanceIdentityDocument()
	if err != nil {
		panic(err)
	}
	src, err := config.FromEC2(cfg, doc.InstanceID, ec2.New(cp))
	if err != nil {
		panic(err)
	}
	return src
}

func createS3Source(cfg sources.Settings, cp client.ConfigProvider) sources.RefreshableSource {
	src, err := config.FromS3(cfg, s3.New(cp))
	switch awsutil.GetCode(err) {
	case "NoCredentialProviders":
		return nil
	case s3.ErrCodeNoSuchBucket: // fake-s3 sadness.
		return nil
	default:
	}
	if err == sources.ErrMissingS3ConfigurationBucket {
		return nil
	}
	if err != nil {
		panic(err)
	}
	return src
}

func createFileSource(cfg sources.Settings, cp client.ConfigProvider) sources.RefreshableSource {
	if env, ok := cfg.TryGetString("app.env"); !ok || strings.HasPrefix(env, "prod") {
		return nil
	}
	src, err := config.FromFilesystem(cfg, localDebugPath)
	if err == sources.ErrMissingFile {
		log.Printf("Local debug file %s not found, skipping filesystem config", localDebugPath)
		return nil
	}
	if err != nil {
		panic(err)
	}
	return src
}

func (d *StandardDefaults) scheduleUpdates(cfg config.Config) config.RefreshController {
	ctrl := config.NewRefreshController()
	for _, s := range d.Sources {
		if s.source != nil {
			name := string(s.Type)
			if s.Type == Custom {
				name = s.Name
			}
			ctrl.Schedule(s.source, cfg, fmt.Sprintf("config.%s.refresh", name), s.Refresh)
		}

	}
	return ctrl
}
