package sources

import (
	"encoding/json"
	"errors"
	"io"
	"os"

	"code.justin.tv/gds/gds/golibs/config/types"
)

var ErrMissingFile = errors.New("Could not find local file")

const filesystemRefreshLogicName = "file: "

type file interface {
	io.ReadCloser
	Name() string
	Stat() (os.FileInfo, error)
}

type fileSource func(name string) (file, error)

type filesystemRefreshLogic struct {
	path string
	open fileSource
}

// NewFilesystemRefreshLogic constructs a RefreshLogic that pulls configuration from the local filesystem
func NewFilesystemRefreshLogic(path string) RefreshLogic {
	return &filesystemRefreshLogic{path, openOsFile}
}

func (s *filesystemRefreshLogic) GetName() string { return filesystemRefreshLogicName }
func (s *filesystemRefreshLogic) GetValues(settings Settings) (Source, error) {
	if s.path == "" {
		return nil, types.ErrArgumentWasNil
	}

	data := make(map[string]interface{})

	// check the filesystem for the value
	file, err := s.open(s.path)
	if err != nil {
		if os.IsNotExist(err) {
			err = ErrMissingFile
		}
		return nil, err
	}

	stat, err := file.Stat()
	if err != nil {
		return nil, err
	}

	buffer := make([]byte, stat.Size())
	if _, err = file.Read(buffer); err != nil {
		return nil, err
	}

	if err = json.Unmarshal(buffer, &data); err != nil {
		return nil, err
	}

	if err = file.Close(); err != nil {
		return nil, err
	}

	return NewStaticSource(s.GetName()+" "+s.path, data)
}

func openOsFile(name string) (file, error) {
	file, err := os.Open(name)
	return file, err
}
