package types

import "sync"

// TODO : put this wrapper at the Config layer once we have value change notification

// ConversionCache is a utility class that optimizes reads by incorrect type
type ConversionCache struct {
	bools   map[string]*boolValue
	floats  map[string]*floatValue
	ints    map[string]*intValue
	strings map[string]*stringValue
	mutex   sync.RWMutex
}

// NewConversionCache initializes and returns a pointer to a ConversionCache
func NewConversionCache() *ConversionCache {
	return &ConversionCache{
		bools:   make(map[string]*boolValue),
		floats:  make(map[string]*floatValue),
		ints:    make(map[string]*intValue),
		strings: make(map[string]*stringValue),
	}
}

// Clear removes all cached conversions for a key, and should be called if
// the uncached value has changed.
func (c *ConversionCache) Clear(name string) {
	c.mutex.Lock()
	defer c.mutex.Unlock()
	delete(c.bools, name)
	delete(c.ints, name)
	delete(c.floats, name)
	delete(c.strings, name)
}

// ConvertToBool caches any necessary conversion to bool
func (c *ConversionCache) ConvertToBool(name string, value interface{}) (bool, bool) {
	if out, ok := value.(bool); ok {
		return out, true
	}
	c.mutex.RLock()
	out, ok := c.bools[name]
	c.mutex.RUnlock()
	if ok {
		return out.value, out.found
	}
	c.mutex.Lock()
	defer c.mutex.Unlock()
	conv := new(boolValue)
	conv.Set(value)
	c.bools[name] = conv
	return conv.value, conv.found
}

// ConvertToFloat caches any necessary conversion to float64
func (c *ConversionCache) ConvertToFloat(name string, value interface{}) (float64, bool) {
	if out, ok := value.(float64); ok {
		return out, true
	}
	c.mutex.RLock()
	out, ok := c.floats[name]
	c.mutex.RUnlock()
	if ok {
		return out.value, out.found
	}
	c.mutex.Lock()
	defer c.mutex.Unlock()
	conv := new(floatValue)
	conv.Set(value)
	c.floats[name] = conv
	return conv.value, conv.found
}

// ConvertToInt caches any necessary conversion to int64
func (c *ConversionCache) ConvertToInt(name string, value interface{}) (int64, bool) {
	if out, ok := value.(int64); ok {
		return out, true
	}
	c.mutex.RLock()
	out, ok := c.ints[name]
	c.mutex.RUnlock()
	if ok {
		return out.value, out.found
	}
	c.mutex.Lock()
	defer c.mutex.Unlock()
	conv := new(intValue)
	conv.Set(value)
	c.ints[name] = conv
	return conv.value, conv.found
}

// ConvertToString caches any necessary conversion to string
func (c *ConversionCache) ConvertToString(name string, value interface{}) (string, bool) {
	if out, ok := value.(string); ok {
		return out, true
	}
	c.mutex.RLock()
	out, ok := c.strings[name]
	c.mutex.RUnlock()
	if ok {
		return out.value, out.found
	}
	c.mutex.Lock()
	defer c.mutex.Unlock()
	conv := new(stringValue)
	conv.Set(value)
	c.strings[name] = conv
	return conv.value, conv.found
}
