package discovery

import (
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/aws/aws-sdk-go/service/sns/snsiface"
)

// SNSTopics maps ARNs to attributes for query
type SNSTopics map[string]*sns.GetTopicAttributesOutput

// Topics returns an SNSTopics object to allow multiple queries on one set of
// service reads.
func Topics(svc snsiface.SNSAPI) (SNSTopics, error) {
	resp, err := svc.ListTopics(&sns.ListTopicsInput{})
	if err != nil {
		return nil, err
	}

	topics := make(SNSTopics)
	for _, topic := range resp.Topics {
		attrResp, err := svc.GetTopicAttributes(&sns.GetTopicAttributesInput{
			TopicArn: topic.TopicArn,
		})
		if err != nil {
			return nil, err
		}
		topics[*topic.TopicArn] = attrResp
	}
	return topics, nil
}

// ByDisplayName attempts to find an ARN by an exact match of display name
func (s SNSTopics) ByDisplayName(displayName string) string {
	for arn, attr := range s {
		if name, ok := attr.Attributes["DisplayName"]; ok && name != nil && *name == displayName {
			return arn
		}
	}
	return ""
}
