package aggregator

import (
	"sync"
	"time"

	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
)

type Read struct {
	db         dynamodbiface.DynamoDBAPI
	heartbeat  time.Duration
	items      []*readSet
	cancel     *time.Timer
	opname     string
	handler    ReadHandler
	errNotRead error
	mutex      sync.Mutex
}

func NewRead(db dynamodbiface.DynamoDBAPI, heartbeat time.Duration) *Read {
	return &Read{
		db:         db,
		heartbeat:  heartbeat,
		handler:    new(emptyHandler),
		errNotRead: ErrNotRead,
	}
}

func (r *Read) SetMetrics(opname string, handler ReadHandler) *Read {
	defer r.mutex.Unlock()
	r.mutex.Lock()
	r.opname = opname
	r.handler = handler
	return r
}

func (r *Read) SetErrNotRead(err error) *Read {
	r.errNotRead = err
	return r
}

func (r *Read) Add(table string, consistent bool, key map[string]*dynamodb.AttributeValue) Promise {
	defer r.mutex.Unlock()
	r.mutex.Lock()
	for _, item := range r.items {
		promise, ok := item.push(table, consistent, key)
		if ok {
			return promise
		}
	}
	item := newReadSet(r)
	r.items = append(r.items, item)
	promise, _ := item.push(table, consistent, key)
	if r.cancel == nil {
		r.cancel = time.AfterFunc(r.heartbeat, r.run)
	}
	return promise
}

func (r *Read) Close() {
	unfinished := r.execute(r.reset())
	for _, u := range unfinished {
		u.cancel()
	}
}

func (r *Read) run() {
	// written this way so we don't lock the`
	// mutex during DB execution
	r.append(r.execute(r.reset()))
}

func (r *Read) execute(items []*readSet) []*readSet {
	out := []*readSet{}
	for _, item := range items {
		o := item.execute()
		if o != nil {
			out = append(out, o)
		}
	}
	return out
}

func (r *Read) reset() []*readSet {
	defer r.mutex.Unlock()
	r.mutex.Lock()
	if r.cancel != nil {
		r.cancel.Stop()
		r.cancel = nil
	}
	out := r.items
	r.items = nil
	return out
}

func (r *Read) append(in []*readSet) {
	defer r.mutex.Unlock()
	r.mutex.Lock()
	for _, item := range in {
		r.items = append(r.items, item)
		if r.cancel == nil {
			r.cancel = time.AfterFunc(r.heartbeat, r.run)
		}
	}
}
