package aggregator

import (
	"errors"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

const bulkLimit = 100

var ErrIllegalKey = errors.New("Illegal key specified")
var ErrNotRead = errors.New("No database response was given")

// NOTE: this is a struct for ReadAggregator use; thread safety concerns
// are handled at that layer
type readSet struct {
	parent     *Read
	tagGen     map[string][]string
	keys       map[string][]map[string]*dynamodb.AttributeValue
	consistent map[string]bool
	promises   map[string]*ReadPromise
}

func newReadSet(parent *Read) *readSet {
	return &readSet{
		parent:     parent,
		tagGen:     make(map[string][]string),
		keys:       make(map[string][]map[string]*dynamodb.AttributeValue),
		consistent: make(map[string]bool),
		promises:   make(map[string]*ReadPromise),
	}
}

func (r *readSet) push(table string, consistent bool, key map[string]*dynamodb.AttributeValue) (Promise, bool) {
	tag, err := r.createTag(table, key)
	if err != nil {
		return &ErrorPromise{err}, true
	}
	if consistent {
		r.consistent[table] = true
	}
	if prev, ok := r.promises[tag]; ok {
		return prev, true // already had a request for this key
	}
	if len(r.promises) == bulkLimit {
		return nil, false // too many requests, need another set
	}
	promise := newPromise(tag)
	r.promises[tag] = promise
	r.keys[table] = append(r.keys[table], key)
	return promise, true
}

func (r *readSet) cancel() {
	for _, p := range r.promises {
		p.set(nil, r.parent.errNotRead)
	}
}

// returns a remainder with original promises intact if the size limit was exceeded
func (r *readSet) execute() *readSet {
	start := time.Now()
	input := new(dynamodb.BatchGetItemInput)
	input.SetReturnConsumedCapacity(r.parent.handler.ReturnConsumedCapacity())
	items := make(map[string]*dynamodb.KeysAndAttributes)
	for table, keys := range r.keys {
		c, ok := r.consistent[table]
		items[table] = &dynamodb.KeysAndAttributes{
			ConsistentRead: aws.Bool(c && ok),
			Keys:           keys,
		}
	}
	input.SetRequestItems(items)
	out, err := r.parent.db.BatchGetItem(input)
	if out != nil {
		r.parent.handler.OnRead(r.parent.opname, time.Since(start), out.ConsumedCapacity...)
	}

	if err != nil {
		for _, p := range r.promises {
			p.set(nil, err)
		}
		return nil
	}

	// handle completed records
	for table, items := range out.Responses {
		for _, item := range items {
			tag, err := r.createTag(table, item)
			if err == nil {
				if p, ok := r.promises[tag]; ok {
					p.set(item, nil)
					delete(r.promises, tag)
				}
			}
		}
	}

	if len(out.UnprocessedKeys) == 0 {
		r.cancel() // handle promises that weren't found
		return nil // we're done, return no work
	}

	// handle incomplete records (request too large), return remainder
	keys := make(map[string][]map[string]*dynamodb.AttributeValue)
	for table, atts := range out.UnprocessedKeys {
		keys[table] = atts.Keys
	}
	return &readSet{
		parent:     r.parent,
		tagGen:     r.tagGen,
		keys:       keys,
		consistent: r.consistent,
		promises:   r.promises,
	}
}

func (r *readSet) createTag(table string, key map[string]*dynamodb.AttributeValue) (string, error) {
	segments := []string{table}
	gen, ok := r.tagGen[table]
	if !ok {
		gen = make([]string, 0, len(key))
		for k := range key {
			gen = append(gen, k)
		}
		r.tagGen[table] = gen
	}
	for _, name := range gen {
		segment, ok := key[name]
		if !ok {
			return "", ErrIllegalKey
		}
		segments = append(segments, segment.GoString())
	}
	return strings.Join(segments, "/"), nil
}
