package dynamodb

import "github.com/aws/aws-sdk-go/service/dynamodb"

// BoolExists checks for a non-nil boolean value at the given key in a given map and returns
// true if it's found and false otherwise
func BoolExists(data map[string]*dynamodb.AttributeValue, key string) bool {
	if data[key] == nil || data[key].BOOL == nil {
		return false
	}
	return true
}

// ListExists checks for a non-nil string at the given key in a given map and returns
// true if it's found and false otherwise
func ListExists(data map[string]*dynamodb.AttributeValue, key string) bool {
	if data[key] == nil || data[key].L == nil {
		return false
	}
	return true
}

// MapExists checks for a non-nil map at the given key in a given map and returns
// true if it's found and false otherwise
func MapExists(data map[string]*dynamodb.AttributeValue, key string) bool {
	if data[key] == nil || data[key].M == nil {
		return false
	}
	return true
}

// NumberExists checks for a non-nil number at the given key in a given map and returns
// true if it's found and false otherwise
func NumberExists(data map[string]*dynamodb.AttributeValue, key string) bool {
	if data[key] == nil || data[key].N == nil {
		return false
	}
	return true
}

// StrExists checks for a non-nil string at the given key in a given map and returns
// true if it's found and false otherwise
func StrExists(data map[string]*dynamodb.AttributeValue, key string) bool {
	if data[key] == nil || data[key].S == nil {
		return false
	}
	return true
}

// NestedStringExists returns true after traversing a map[string]*dynamodb.AttributeValue tree
// through the entire given path and finds a String AttributeValue. false is returned if nil is found
// at any point in the path
func NestedStringExists(data map[string]*dynamodb.AttributeValue, path ...string) bool {
	dirLength := len(path) - 1
	if dirLength < 0 {
		return false
	}

	lastLeaf := GetSubmap(data, path[:dirLength]...)
	if lastLeaf == nil {
		return false
	}

	return StrExists(lastLeaf, path[dirLength])
}

// NestedBoolExists returns true after traversing a map[string]*dynamodb.AttributeValue tree
// through the entire given path and finds a Bool AttributeValue. false is returned if nil is found
// at any point in the path
func NestedBoolExists(data map[string]*dynamodb.AttributeValue, path ...string) bool {
	dirLength := len(path) - 1
	if dirLength < 0 {
		return false
	}

	lastLeaf := GetSubmap(data, path[:dirLength]...)
	if lastLeaf == nil {
		return false
	}

	return BoolExists(lastLeaf, path[dirLength])
}

// NestedListExists returns true after traversing a map[string]*dynamodb.AttributeValue tree
// through the entire given path and finds a List AttributeValue. false is returned if nil is found
// at any point in the path
func NestedListExists(data map[string]*dynamodb.AttributeValue, path ...string) bool {
	dirLength := len(path) - 1
	if dirLength < 0 {
		return false
	}

	lastLeaf := GetSubmap(data, path[:dirLength]...)
	if lastLeaf == nil {
		return false
	}

	return ListExists(lastLeaf, path[dirLength])
}

// NestedMapExists returns true after traversing a map[string]*dynamodb.AttributeValue tree
// through the entire given path and finds a Map AttributeValue. false is returned if nil is found
// at any point in the path
func NestedMapExists(data map[string]*dynamodb.AttributeValue, path ...string) bool {
	dirLength := len(path) - 1
	if dirLength < 0 {
		return false
	}

	lastLeaf := GetSubmap(data, path[:dirLength]...)
	if lastLeaf == nil {
		return false
	}

	return MapExists(lastLeaf, path[dirLength])
}

// NestedNumberExists returns true after traversing a map[string]*dynamodb.AttributeValue tree
// through the entire given path and finds a Bool AttributeValue. false is returned if nil is found
// at any point in the path
func NestedNumberExists(data map[string]*dynamodb.AttributeValue, path ...string) bool {
	dirLength := len(path) - 1
	if dirLength < 0 {
		return false
	}

	lastLeaf := GetSubmap(data, path[:dirLength]...)
	if lastLeaf == nil {
		return false
	}

	return NumberExists(lastLeaf, path[dirLength])
}

// GetSubmap attempts to walk a list of map attributes along the specified path; it returns nil
// if the map could not be found.
func GetSubmap(data map[string]*dynamodb.AttributeValue, path ...string) map[string]*dynamodb.AttributeValue {
	currentData := data
	for i := 0; i < len(path); i++ {
		currentKey := path[i]

		if currentData[currentKey] != nil && currentData[currentKey].M != nil {
			currentData = currentData[currentKey].M
		} else {
			return nil
		}
	}

	return currentData
}
